<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once( dirname( __FILE__ ) . '/nuvei_tps_xml.php' );

/**
 * WC_Gateway_Nuvei class.
 *
 * @extends WC_Payment_Gateway
 */
class WC_Gateway_Nuvei extends WC_Payment_Gateway_CC {

	/**
	 * Should we capture Credit cards
	 *
	 * @var bool
	 */
	public $capture;

	/**
	 * Alternate credit card statement name
	 *
	 * @var bool
	 */
	public $statement_descriptor;

	/**
	 * Checkout enabled
	 *
	 * @var bool
	 */
	public $nuvei_checkout;

	/**
	 * Checkout Locale
	 *
	 * @var string
	 */
	public $nuvei_checkout_locale;

	/**
	 * Credit card image
	 *
	 * @var string
	 */
	public $nuvei_checkout_image;

	/**
	 * Should we store the users credit cards?
	 *
	 * @var bool
	 */
	public $saved_cards;

	/**
	 * API access secret key
	 *
	 * @var string
	 */
	public $secret_key;

	/**
	 * Api access publishable key
	 *
	 * @var string
	 */
	public $publishable_key;

	/**
	 * Is test mode active?
	 *
	 * @var bool
	 */
	public $testmode;

	/**
	 * Logging enabled?
	 *
	 * @var bool
	 */
	public $logging;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->id                   = 'nuvei';
		$this->method_title         = __( 'Nuvei', 'woocommerce-gateway-nuvei' );
		$this->method_description   = sprintf( __( 'Nuvei works by adding credit card fields on the checkout and then sending the details to Nuvei for verification.', 'woocommerce-gateway-nuvei' ));
		$this->has_fields           = true;
		$this->supports             = array(
			//'subscriptions',
			'products',
			//'refunds',
			//'subscription_cancellation',
			//'subscription_reactivation',
			//'subscription_suspension',
			//'subscription_amount_changes',
			//'subscription_payment_method_change', // Subs 1.n compatibility.
			//'subscription_payment_method_change_customer',
			//'subscription_payment_method_change_admin',
			//'subscription_date_changes',
			//'multiple_subscriptions',
			'pre-orders',
			//'tokenization',
			'add_payment_method',
		);

		// Load the form fields.
		$this->init_form_fields();

		// Load the settings.
		$this->init_settings();

		// Get setting values.
		$this->title                   = $this->get_option( 'title' );
		$this->description             = $this->get_option( 'description' );
		$this->enabled                 = $this->get_option( 'enabled' );
		$this->testmode                = 'yes' === $this->get_option( 'testmode' );
		$this->capture                 = 'yes' === $this->get_option( 'capture', 'yes' );
		$this->statement_descriptor    = $this->get_option( 'statement_descriptor', wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES ) );
		$this->nuvei_checkout         = 'yes' === $this->get_option( 'nuvei_checkout' );
		$this->nuvei_integration_type = $this->get_option( 'nuvei_integration_type' );
		$this->nuvei_checkout_locale  = $this->get_option( 'nuvei_checkout_locale' );
		$this->nuvei_checkout_image   = $this->get_option( 'nuvei_checkout_image', '' );
		$this->saved_cards             = 'yes' === $this->get_option( 'saved_cards' );
		$this->secret_key              = $this->testmode ? $this->get_option( 'test_secret_key' ) : $this->get_option( 'secret_key' );
		$this->publishable_key         = $this->testmode ? $this->get_option( 'test_publishable_key' ) : $this->get_option( 'publishable_key' );
		$this->currency         = $this->testmode ? $this->get_option( 'test_currency' ) : $this->get_option( 'currency' );
		$this->multicurrency         = $this->testmode ? $this->get_option( 'test_multicurrency' ) : $this->get_option( 'multicurrency' );
		$this->secret_key2              = $this->testmode ? $this->get_option( 'test_secret_key2' ) : $this->get_option( 'secret_key2' );
		$this->publishable_key2         = $this->testmode ? $this->get_option( 'test_publishable_key2' ) : $this->get_option( 'publishable_key2' );
		$this->currency2         = $this->testmode ? $this->get_option( 'test_currency2' ) : $this->get_option( 'currency2' );
		$this->multicurrency2         = $this->testmode ? $this->get_option( 'test_multicurrency2' ) : $this->get_option( 'multicurrency2' );
		$this->secret_key3              = $this->testmode ? $this->get_option( 'test_secret_key3' ) : $this->get_option( 'secret_key3' );
		$this->publishable_key3         = $this->testmode ? $this->get_option( 'test_publishable_key3' ) : $this->get_option( 'publishable_key3' );
		$this->currency3         = $this->testmode ? $this->get_option( 'test_currency3' ) : $this->get_option( 'currency3' );
		$this->multicurrency3         = $this->testmode ? $this->get_option( 'test_multicurrency3' ) : $this->get_option( 'multicurrency3' );
		$this->logging                 = 'yes' === $this->get_option( 'logging' );

		if ( $this->nuvei_integration_type=='hpp') {
			$this->order_button_text = __( 'Continue to payment', 'woocommerce-gateway-nuvei' );
		}

		if ( $this->testmode ) {
			$this->description .= ' ' . sprintf( __( ' - TEST MODE ENABLED') );
			$this->description  = trim( $this->description );
		}

		// Hooks.
		add_action( 'wp_enqueue_scripts', array( $this, 'payment_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
	}


    /**
     * Outputs fields for entering credit card information.
     * @since 2.6.0
     */
    public function form() {
        wp_enqueue_script( 'wc-credit-card-form' );

        $fields = array();

        $cvc_field = '<p class="form-row form-row-last">
			<label for="' . esc_attr( $this->id ) . '-card-cvc">' . esc_html__( 'Card code', 'woocommerce' ) . ' <span class="required">*</span></label>
			<input id="' . esc_attr( $this->id ) . '-card-cvc" class="input-text wc-credit-card-form-card-cvc" inputmode="numeric" autocomplete="off" autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" maxlength="4" placeholder="' . esc_attr__( 'CVC', 'woocommerce' ) . '" ' . $this->field_name( 'card-cvc' ) . ' style="width:100px" />
		</p>';

        $default_fields = array(
            'card-name-field' => '<p class="form-row form-row-wide">
				<label for="' . esc_attr( $this->id ) . '-card-name">' . esc_html__( 'Card holder', 'woocommerce' ) . ' <span class="required">*</span></label>
				<input id="' . esc_attr( $this->id ) . '-card-name" class="input-text wc-credit-card-form-card-name" inputmode="text" autocomplete="cc-name" autocorrect="no" autocapitalize="no" spellcheck="no" type="text" placeholder="" ' . $this->field_name( 'card-name' ) . ' />
			</p>',
            'card-number-field' => '<p class="form-row form-row-wide">
				<label for="' . esc_attr( $this->id ) . '-card-number">' . esc_html__( 'Card number', 'woocommerce' ) . ' <span class="required">*</span></label>
				<input id="' . esc_attr( $this->id ) . '-card-number" class="input-text wc-credit-card-form-card-number" inputmode="numeric" autocomplete="cc-number" autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" placeholder="&bull;&bull;&bull;&bull; &bull;&bull;&bull;&bull; &bull;&bull;&bull;&bull; &bull;&bull;&bull;&bull;" ' . $this->field_name( 'card-number' ) . ' />
			</p>',
            'card-expiry-field' => '<p class="form-row form-row-first">
				<label for="' . esc_attr( $this->id ) . '-card-expiry">' . esc_html__( 'Expiry (MM/YY)', 'woocommerce' ) . ' <span class="required">*</span></label>
				<input id="' . esc_attr( $this->id ) . '-card-expiry" class="input-text wc-credit-card-form-card-expiry" inputmode="numeric" autocomplete="cc-exp" autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" placeholder="' . esc_attr__( 'MM / YY', 'woocommerce' ) . '" ' . $this->field_name( 'card-expiry' ) . ' />
			</p>',
        );

        if ( ! $this->supports( 'credit_card_form_cvc_on_saved_method' ) ) {
            $default_fields['card-cvc-field'] = $cvc_field;
        }

        $fields = wp_parse_args( $fields, apply_filters( 'woocommerce_credit_card_form_fields', $default_fields, $this->id ) );
        ?>

        <fieldset id="wc-<?php echo esc_attr( $this->id ); ?>-cc-form" class='wc-credit-card-form wc-payment-form'>
            <?php do_action( 'woocommerce_credit_card_form_start', $this->id ); ?>
            <?php
            foreach ( $fields as $field ) {
                echo $field;
            }
            ?>
            <?php do_action( 'woocommerce_credit_card_form_end', $this->id ); ?>
            <div class="clear"></div>
        </fieldset>
        <?php

        if ( $this->supports( 'credit_card_form_cvc_on_saved_method' ) ) {
            echo '<fieldset>' . $cvc_field . '</fieldset>';
        }
    }

	/**
	 * Get_icon function.
	 *
	 * @access public
	 * @return string
	 */
	public function get_icon() {
		$ext   = version_compare( WC()->version, '2.6', '>=' ) ? '.svg' : '.png';
		$style = version_compare( WC()->version, '2.6', '>=' ) ? 'style="margin-left: 0.3em"' : '';

		$icon  = '<img src="' . WC_HTTPS::force_https_url( WC()->plugin_url() . '/assets/images/icons/credit-cards/visa' . $ext ) . '" alt="Visa" width="32" ' . $style . ' />';
		$icon .= '<img src="' . WC_HTTPS::force_https_url( WC()->plugin_url() . '/assets/images/icons/credit-cards/mastercard' . $ext ) . '" alt="Mastercard" width="32" ' . $style . ' />';
		$icon .= '<img src="' . WC_HTTPS::force_https_url( WC()->plugin_url() . '/assets/images/icons/credit-cards/amex' . $ext ) . '" alt="Amex" width="32" ' . $style . ' />';

		if ( 'USD' === get_woocommerce_currency() ) {
			$icon .= '<img src="' . WC_HTTPS::force_https_url( WC()->plugin_url() . '/assets/images/icons/credit-cards/discover' . $ext ) . '" alt="Discover" width="32" ' . $style . ' />';
			$icon .= '<img src="' . WC_HTTPS::force_https_url( WC()->plugin_url() . '/assets/images/icons/credit-cards/jcb' . $ext ) . '" alt="JCB" width="32" ' . $style . ' />';
			$icon .= '<img src="' . WC_HTTPS::force_https_url( WC()->plugin_url() . '/assets/images/icons/credit-cards/diners' . $ext ) . '" alt="Diners" width="32" ' . $style . ' />';
		}

		return apply_filters( 'woocommerce_gateway_icon', $icon, $this->id );
	}

	/**
	 * Get Nuvei amount to pay
	 *
	 * @param float  $total Amount due.
	 * @param string $currency Accepted currency.
	 *
	 * @return float|int
	 */
	public function get_nuvei_amount( $total, $currency = '' ) {
		if ( ! $currency ) {
			$currency = get_woocommerce_currency();
		}
		switch ( strtoupper( $currency ) ) {
			// Zero decimal currencies.
			case 'BIF' :
			case 'CLP' :
			case 'DJF' :
			case 'GNF' :
			case 'JPY' :
			case 'KMF' :
			case 'KRW' :
			case 'MGA' :
			case 'PYG' :
			case 'RWF' :
			case 'VND' :
			case 'VUV' :
			case 'XAF' :
			case 'XOF' :
			case 'XPF' :
				$total = absint( $total );
				break;
			default :
				$total = round( $total, 2 ) * 100; // In cents.
				break;
		}
		return $total;
	}

	/**
	 * Check if SSL is enabled and notify the user
	 */
	public function admin_notices() {
		if ( 'no' === $this->enabled ) {
			return;
		}

		// Show message if enabled and FORCE SSL is disabled and WordpressHTTPS plugin is not detected.
		if ( ( function_exists( 'wc_site_is_https' ) && ! wc_site_is_https() ) && ( 'no' === get_option( 'woocommerce_force_ssl_checkout' ) && ! class_exists( 'WordPressHTTPS' ) ) ) {
			echo '<div class="error nuvei-ssl-message"><p>' . sprintf( __( 'Nuvei is enabled, but the <a href="%1$s">force SSL option</a> is disabled; your checkout may not be secure! Please enable SSL and ensure your server has a valid <a href="%2$s" target="_blank">SSL certificate</a> - Nuvei will only work in test mode.', 'woocommerce-gateway-nuvei' ), admin_url( 'admin.php?page=wc-settings&tab=checkout' ), 'https://en.wikipedia.org/wiki/Transport_Layer_Security' ) . '</p></div>';
		}
	}

	/**
	 * Check if this gateway is enabled
	 */
	public function is_available() {
		if ( 'yes' === $this->enabled ) {
			if ( ! $this->testmode && is_checkout() && ! is_ssl() ) {
				return false;
			}
			if ( (! $this->secret_key || ! $this->publishable_key ) && (! $this->secret_key2 || ! $this->publishable_key2 ) && (! $this->secret_key3 || ! $this->publishable_key3 ))  {
				return false;
			}
			return true;
		}
		return false;
	}

	/**
	 * Initialise Gateway Settings Form Fields
	 */
	public function init_form_fields() {
		$this->form_fields = include( 'settings-nuvei.php' );
	}

	/**
	 * Payment form on checkout page
	 */
	public function payment_fields() {
		$user                 = wp_get_current_user();
		$display_tokenization = $this->supports( 'tokenization' ) && is_checkout() && $this->saved_cards;
		$total                = WC()->cart->total;

		// If paying from order, we need to get total from order not cart.
		if ( isset( $_GET['pay_for_order'] ) && ! empty( $_GET['key'] ) ) {
			$order = wc_get_order( wc_get_order_id_by_order_key( wc_clean( $_GET['key'] ) ) );
			$total = $order->get_total();
		}

		if ( $user->ID ) {
			$user_email = get_user_meta( $user->ID, 'billing_email', true );
			$user_email = $user_email ? $user_email : $user->user_email;
		} else {
			$user_email = '';
		}

		if ( is_add_payment_method_page() ) {
			$pay_button_text = __( 'Add Card', 'woocommerce-gateway-nuvei' );
			$total        = '';
		} else {
			$pay_button_text = '';
		}


		echo '<div
			id="nuvei-payment-data"
			data-panel-label="' . esc_attr( $pay_button_text ) . '"
			data-description=""
			data-email="' . esc_attr( $user_email ) . '"
			data-amount="' . esc_attr( $this->get_nuvei_amount( $total )/100 ) . '"
			data-name="' . esc_attr( $this->statement_descriptor ) . '"
			data-currency="' . esc_attr( strtoupper( get_woocommerce_currency() ) ) . '"
			data-image="' . esc_attr( $this->nuvei_checkout_image ) . '"
			data-locale="' . esc_attr( $this->nuvei_checkout_locale ? $this->nuvei_checkout_locale : 'en' ) . '"
			data-allow-remember-me="' . esc_attr( $this->saved_cards ? 'true' : 'false' ) . '"
			data-terminal-id="' . esc_attr( $this->publishable_key ) . '">';

		if ( $this->description ) {
			echo apply_filters( 'wc_nuvei_description', wpautop( wp_kses_post( $this->description ) ) );
		}

		if ( $display_tokenization ) {
			$this->tokenization_script();
			$this->saved_payment_methods();
		}

		if ( $this->nuvei_integration_type=='xml') {
			$this->form();

			if ( apply_filters( 'wc_nuvei_display_save_payment_method_checkbox', $display_tokenization ) ) {
				$this->save_payment_method_checkbox();
			}
		}

		echo '</div>';
	}

	/**
	 * Localize Nuvei messages based on code
	 *
	 * @since 3.0.6
	 * @version 3.0.6
	 * @return array
	 */
	public function get_localized_messages() {
		return apply_filters( 'wc_nuvei_localized_messages', array(
			'invalid_number'        => __( 'The card number is not a valid credit card number.', 'woocommerce-gateway-nuvei' ),
			'invalid_expiry_month'  => __( 'The card\'s expiration month is invalid.', 'woocommerce-gateway-nuvei' ),
			'invalid_expiry_year'   => __( 'The card\'s expiration year is invalid.', 'woocommerce-gateway-nuvei' ),
			'invalid_cvc'           => __( 'The card\'s security code is invalid.', 'woocommerce-gateway-nuvei' ),
			'incorrect_number'      => __( 'The card number is incorrect.', 'woocommerce-gateway-nuvei' ),
			'expired_card'          => __( 'The card has expired.', 'woocommerce-gateway-nuvei' ),
			'incorrect_cvc'         => __( 'The card\'s security code is incorrect.', 'woocommerce-gateway-nuvei' ),
			'incorrect_zip'         => __( 'The card\'s zip code failed validation.', 'woocommerce-gateway-nuvei' ),
			'card_declined'         => __( 'The card was declined.', 'woocommerce-gateway-nuvei' ),
			'missing'               => __( 'There is no card on a customer that is being charged.', 'woocommerce-gateway-nuvei' ),
			'processing_error'      => __( 'An error occurred while processing the card.', 'woocommerce-gateway-nuvei' ),
			'invalid_request_error' => __( 'Could not find payment information.', 'woocommerce-gateway-nuvei' ),
		) );
	}

	/**
	 * Load admin scripts.
	 *
	 * @since 3.1.0
	 * @version 3.1.0
	 */
	public function admin_scripts() {
		if ( 'woocommerce_page_wc-settings' !== get_current_screen()->id ) {
			return;
		}


		wp_enqueue_script( 'woocommerce_nuvei_admin', plugins_url( 'assets/js/nuvei-admin.js', WC_Nuvei_MAIN_FILE ), array(), WC_Nuvei_VERSION, true );

		$nuvei_admin_params = array(
			'localized_messages' => array(
				'not_valid_live_key_msg' => __( 'This is not a valid live key. Live keys start with "sk_live_" and "pk_live_".', 'woocommerce-gateway-nuvei' ),
				'not_valid_test_key_msg' => __( 'This is not a valid test key. Test keys start with "sk_test_" and "pk_test_".', 'woocommerce-gateway-nuvei' ),
				're_verify_button_text'  => __( 'Re-verify Domain', 'woocommerce-gateway-nuvei' ),
				'missing_secret_key'     => __( 'Missing Secret Key. Please set the secret key field above and re-try.', 'woocommerce-gateway-nuvei' ),
			),
			'ajaxurl'            => admin_url( 'admin-ajax.php' )
		);

		wp_localize_script( 'woocommerce_nuvei_admin', 'wc_nuvei_admin_params', apply_filters( 'wc_nuvei_admin_params', $nuvei_admin_params ) );
	}

	/**
	 * payment_scripts function.
	 *
	 * Outputs scripts used for nuvei payment
	 *
	 * @access public
	 */
	public function payment_scripts() {
		if ( ! is_cart() && ! is_checkout() && ! isset( $_GET['pay_for_order'] ) && ! is_add_payment_method_page() ) {
			return;
		}

		if ( $this->nuvei_integration_type == 'hpp' ) {
			wp_enqueue_script( 'woocommerce_nuvei', plugins_url( 'assets/js/nuvei-hpp.js', WC_Nuvei_MAIN_FILE ), '', WC_Nuvei_VERSION, true );
		} else {
			wp_enqueue_script( 'ccValidator', plugins_url( 'assets/js/jquery.creditCardValidator.js', WC_Nuvei_MAIN_FILE ), '', WC_Nuvei_VERSION, true );
			wp_enqueue_script( 'woocommerce_nuvei', plugins_url( 'assets/js/nuvei-xml.js', WC_Nuvei_MAIN_FILE ), array( 'jquery-payment', 'ccValidator'), WC_Nuvei_VERSION, true );
		}

		$nuvei_params = array(
			'key'                  => $this->publishable_key,
			'i18n_terms'           => __( 'Please accept the terms and conditions first', 'woocommerce-gateway-nuvei' ),
			'i18n_required_fields' => __( 'Please fill in required checkout fields first', 'woocommerce-gateway-nuvei' ),
		);

		// If we're on the pay page we need to pass nuvei.js the address of the order.
		if ( isset( $_GET['pay_for_order'] ) && 'true' === $_GET['pay_for_order'] ) {
			$order_id = wc_get_order_id_by_order_key( urldecode( $_GET['key'] ) );
			$order    = wc_get_order( $order_id );

			$nuvei_params['billing_first_name'] = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_first_name : $order->get_billing_first_name();
			$nuvei_params['billing_last_name']  = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_last_name : $order->get_billing_last_name();
			$nuvei_params['billing_address_1']  = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_1 : $order->get_billing_address_1();
			$nuvei_params['billing_address_2']  = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_2 : $order->get_billing_address_2();
			$nuvei_params['billing_state']      = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_state : $order->get_billing_state();
			$nuvei_params['billing_city']       = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_city : $order->get_billing_city();
			$nuvei_params['billing_postcode']   = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_postcode : $order->get_billing_postcode();
			$nuvei_params['billing_country']    = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_country : $order->get_billing_country();
		}

		$nuvei_params['no_prepaid_card_msg']                     = __( 'Sorry, we\'re not accepting prepaid cards at this time.', 'woocommerce-gateway-nuvei' );
		$nuvei_params['allow_prepaid_card']                      = apply_filters( 'wc_nuvei_allow_prepaid_card', true ) ? 'yes' : 'no';
		$nuvei_params['nuvei_checkout_require_billing_address'] = apply_filters( 'wc_nuvei_checkout_require_billing_address', false ) ? 'yes' : 'no';

		// merge localized messages to be use in JS
		$nuvei_params = array_merge( $nuvei_params, $this->get_localized_messages() );

		wp_localize_script( 'woocommerce_nuvei', 'wc_nuvei_params', apply_filters( 'wc_nuvei_params', $nuvei_params ) );
	}

    public function getCreditCardType($str)
    {
        if (empty($str)) {
            return false;
        }

        $matchingPatterns = [
            'visa' => '/^4[0-9]{12}(?:[0-9]{3})?$/',
            'mastercard' => '/^5[1-5][0-9]{14}$/',
            'amex' => '/^3[47][0-9]{13}$/',
            'diners' => '/^3(?:0[0-5]|[68][0-9])[0-9]{11}$/',
            'discover' => '/^6(?:011|5[0-9]{2})[0-9]{12}$/',
            'jcb' => '/^(?:2131|1800|35\d{3})\d{11}$/',
            'any' => '/^(?:4[0-9]{12}(?:[0-9]{3})?|5[1-5][0-9]{14}|6(?:011|5[0-9][0-9])[0-9]{12}|3[47][0-9]{13}|3(?:0[0-5]|[68][0-9])[0-9]{11}|(?:2131|1800|35\d{3})\d{11})$/'
        ];

        foreach ($matchingPatterns as $key=>$pattern) {
            if (preg_match($pattern, $str)) {
                return strtoupper($key);
            }
        }
    }

	/**
	 * Process the payment
	 *
	 * @param int  $order_id Reference.
	 * @param bool $retry Should we retry on fail.
	 * @param bool $force_customer Force user creation.
	 *
	 * @throws Exception If payment will not be accepted.
	 *
	 * @return array|void
	 */
	public function process_payment( $order_id, $retry = true, $force_customer = false ) {
		if ( $this->nuvei_integration_type=='hpp') {
			$order = wc_get_order($order_id);

			$dateTime = date('d-n-Y:H:i:s:v');
			$receiptPageUrl = $this->get_return_url($order);

			$currency = $_POST['currency'];
			if($currency == $this->currency2 && $this->publishable_key2 && $this->secret_key2) {
				$terminalId = $this->publishable_key2;
				$secret = $this->secret_key2;
				$multicurrency = $this->multicurrency2;
			} else if($currency == $this->currency3 && $this->publishable_key3 && $this->secret_key3) {
					$terminalId = $this->publishable_key3;
					$secret = $this->secret_key3;
					$multicurrency = $this->multicurrency3;
			} else {
				$terminalId = $this->publishable_key;
				$secret = $this->secret_key;
				$multicurrency = $this->multicurrency;
			}

			$hash = md5($terminalId . $order_id . ($multicurrency=='yes' ? $currency : '') . number_format($_POST['amount'], 2, '.', '') . $dateTime . $receiptPageUrl . $secret);

			$redirectUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/paymentpage':'https://payments.nuvei.com/merchant/paymentpage';
			$redirectUrl .= '?TERMINALID=' . $terminalId . '&ORDERID=' . $order_id . '&CURRENCY=' . $_POST['currency'] . '&AMOUNT=' . number_format($_POST['amount'], 2, '.', '') . '&DATETIME=' . $dateTime . '&RECEIPTPAGEURL=' . urlencode($receiptPageUrl) . '&HASH=' . $hash;

			return array(
				'result' => 'success',
				'redirect' => $redirectUrl,
			);
		} else if ( $this->nuvei_integration_type=='xml') {
			$order = wc_get_order($order_id);

			$currency = $_POST['currency'];
			if($currency == $this->currency2 && $this->publishable_key2 && $this->secret_key2) {
				$terminalId = $this->publishable_key2;
				$secret = $this->secret_key2;
				$multicurrency = $this->multicurrency2;
			} else if($currency == $this->currency3 && $this->publishable_key3 && $this->secret_key3) {
				$terminalId = $this->publishable_key3;
				$secret = $this->secret_key3;
				$multicurrency = $this->multicurrency3;
			} else {
				$terminalId = $this->publishable_key;
				$secret = $this->secret_key;
				$multicurrency = $this->multicurrency;
			}

			# Set up the authorisation object
			$amount = number_format($_POST['amount'], 2, '.', '');
			$cardNumber = str_replace(' ', '', $_POST['card-number']);
			$cardType = $this->getCreditCardType($cardNumber);
			$cardHolder = $_POST['card-name'];
			$cardExpiry = $_POST['card-expiry'];
			$cvv = $_POST['cvc'];

			$adminEmail = '';
			$adminPhone = '';

			$XmlAuthRequest = new XmlAuthRequest($terminalId, $order_id, $currency, $amount, $cardNumber, $cardType);

			if ($cardType != "SECURECARD") $XmlAuthRequest->SetNonSecureCardCardInfo($cardExpiry, $cardHolder);
			if ($cvv != "") $XmlAuthRequest->SetCvv($cvv);
			if ($multicurrency=='yes') $XmlAuthRequest->SetMultiCur();

			# Perform the online authorisation and read in the result
			$serverUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
			$response = $XmlAuthRequest->ProcessRequestToGateway($secret, $serverUrl);

			$expectedResponseHash = md5($terminalId . $response->UniqueRef() . ($multicurrency=='yes' ? $currency : '')  . $amount . $response->DateTime() . $response->ResponseCode() . $response->ResponseText() . $secret);
			$isHashCorrect = ($expectedResponseHash == $response->Hash());

			$nuveiResponse = '';

			if ($response->IsError()) $nuveiResponse .= 'AN ERROR OCCURED! You transaction was not processed. Error details: ' . $response->ErrorString();
			elseif ($isHashCorrect) {
				switch ($response->ResponseCode()) {
					case "A" :    # -- If using local database, update order as Authorised.
						$nuveiResponse .= 'Payment Processed successfully. Thanks you for your order.';
						$uniqueRef = $response->UniqueRef();
						$responseText = $response->ResponseText();
						$approvalCode = $response->ApprovalCode();
						$avsResponse = $response->AvsResponse();
						$cvvResponse = $response->CvvResponse();
						break;
					case "R" :
					case "D" :
					case "C" :
					case "S" :
					default  :    # -- If using local database, update order as declined/failed --
						$nuveiResponse .= 'PAYMENT DECLINED! Please try again with another card. Bank response: ' . $response->ResponseText();
				}
			} else {
				$nuveiResponse .= 'PAYMENT FAILED: INVALID RESPONSE HASH. Please contact ' . $adminEmail . ' or call ' . $adminPhone . ' to clarify if you will get charged for this order.';
				if ($response->UniqueRef()) $nuveiResponse .= 'Please quote Nuvei Terminal ID: ' . $terminalId . ', and Unique Reference: ' . $response->UniqueRef() . ' when mailing or calling.';
			}

			$receiptPageUrl = $this->get_return_url($order);

			if (!$response->IsError() && $isHashCorrect && $response->ResponseCode() == 'A') {
                wc_reduce_stock_levels( $order_id );
				$order->update_status( apply_filters( 'woocommerce_gateway_nuvei_process_payment_order_status', $order->has_downloadable_item() ? 'on-hold' : 'processing', $order ), __( 'Payment successfully processed #'. $response->UniqueRef(), 'woocommerce' ) );

				WC()->mailer()->customer_invoice( $order );

				WC()->cart->empty_cart();

				return array(
					'result' => 'success',
					'redirect' => $receiptPageUrl,
				);
			}
			else {
				wc_add_notice( $nuveiResponse, 'error' );
			}
		}

	}

	/**
     * Sends the failed order email to admin
     *
     * @version 3.1.0
     * @since 3.1.0
     * @param int $order_id
     * @return null
     */
	public function send_failed_order_email( $order_id ) {
        $emails = WC()->mailer()->get_emails();
        if ( ! empty( $emails ) && ! empty( $order_id ) ) {
            $emails['WC_Email_Failed_Order']->trigger( $order_id );
        }
    }

	/**
	 * Logs
	 *
	 * @since 3.1.0
	 * @version 3.1.0
	 *
	 * @param string $message
	 */
	public function log( $message ) {
		if ( $this->logging ) {
			WC_Nuvei::log( $message );
		}
	}
}
