<?php
/*
 * Plugin Name: WooCommerce Worldnet Gateway
 * Description: Take credit card payments on your store using Worldnet.
 * Author: WorldNetTPS
 * Version: 1.6.2
 * Text Domain: woocommerce-gateway-worldnet
 *
 * Copyright (c) 2017 WorldNetTPS
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Required minimums and constants
 */
define( 'WC_Worldnet_VERSION', '1.6.2' );
define( 'WC_Worldnet_MIN_PHP_VER', '5.6.0' );
define( 'WC_Worldnet_MIN_WC_VER', '2.5.0' );
define( 'WC_Worldnet_MAIN_FILE', __FILE__ );

if ( ! class_exists( 'WC_Worldnet' ) ) :

	class WC_Worldnet {

		/**
		 * @var Singleton The reference the *Singleton* instance of this class
		 */
		private static $instance;

		/**
		 * @var Reference to logging class.
		 */
		private static $log;

		/**
		 * Returns the *Singleton* instance of this class.
		 *
		 * @return Singleton The *Singleton* instance.
		 */
		public static function get_instance() {
			if ( null === self::$instance ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Private clone method to prevent cloning of the instance of the
		 * *Singleton* instance.
		 *
		 * @return void
		 */
		private function __clone() {}

		/**
		 * Private unserialize method to prevent unserializing of the *Singleton*
		 * instance.
		 *
		 * @return void
		 */
		private function __wakeup() {}

		/**
		 * Flag to indicate whether or not we need to load code for / support subscriptions.
		 *
		 * @var bool
		 */
		private $subscription_support_enabled = false;

		/**
		 * Flag to indicate whether or not we need to load support for pre-orders.
		 *
		 * @since 3.0.3
		 *
		 * @var bool
		 */
		private $pre_order_enabled = false;

		/**
		 * Notices (array)
		 * @var array
		 */
		public $notices = array();

		/**
		 * Protected constructor to prevent creating a new instance of the
		 * *Singleton* via the `new` operator from outside of this class.
		 */
		protected function __construct() {
			add_action( 'admin_init', array( $this, 'check_environment' ) );
			add_action( 'admin_notices', array( $this, 'admin_notices' ), 15 );
			add_action( 'plugins_loaded', array( $this, 'init' ) );
		}

		/**
		 * Init the plugin after plugins_loaded so environment variables are set.
		 */
		public function init() {
			// Don't hook anything else in the plugin if we're in an incompatible environment
			if ( self::get_environment_warning() ) {
				return;
			}

			// Init the gateway itself
			$this->init_gateways();

			add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );
			add_action( 'woocommerce_order_status_on-hold_to_processing', array( $this, 'capture_payment' ) );
			add_action( 'woocommerce_order_status_on-hold_to_completed', array( $this, 'capture_payment' ) );
			add_action( 'woocommerce_order_status_on-hold_to_cancelled', array( $this, 'cancel_payment' ) );
			add_action( 'woocommerce_order_status_on-hold_to_refunded', array( $this, 'cancel_payment' ) );
			add_filter( 'woocommerce_get_customer_payment_tokens', array( $this, 'woocommerce_get_customer_payment_tokens' ), 10, 3 );
			add_action( 'woocommerce_payment_token_deleted', array( $this, 'woocommerce_payment_token_deleted' ), 10, 2 );
			add_action( 'woocommerce_payment_token_set_default', array( $this, 'woocommerce_payment_token_set_default' ) );
			add_action( 'wp_ajax_worldnet_dismiss_request_api_notice', array( $this, 'dismiss_request_api_notice' ) );

			add_action( 'woocommerce_after_register_post_type', array( $this, 'checkResponse'), 10, 1 );
		}

		public function checkResponse() {

            if (isset($_GET['ORDERID']) && $_GET['ORDERID'] && isset($_GET['HASH']) && $_GET['HASH']) {
                if ($_GET['RESPONSECODE'] == 'A') { //TODO: perform background validation
                    // Lets grab the order
                    $order_id = wc_get_order_id_by_order_key(urldecode($_GET['key']));
                    $order = wc_get_order($order_id);

                    if($order->get_status() == 'pending' ||$order->get_status() == 'failed') {
                        wc_reduce_stock_levels( $order_id );
                        $order->update_status(apply_filters('woocommerce_gateway_worldnet_process_payment_order_status', $order->has_downloadable_item() ? 'on-hold' : 'processing', $order), __('Payment processed correctly #' . $_GET['UNIQUEREF'], 'woocommerce'));

                        self::send_new_order_email($order_id);

                        WC()->mailer()->customer_invoice($order);

                        // Remove cart
                        WC()->cart->empty_cart();
                    }
                } else {
                    $order_id = wc_get_order_id_by_order_key(urldecode($_GET['key']));
                    $order = wc_get_order($order_id);

                    $order->update_status(apply_filters('woocommerce_gateway_worldnet_process_payment_order_status', $order->has_downloadable_item() ? 'on-hold' : 'failed', $order), __('Payment failed #' . $_GET['UNIQUEREF'], 'woocommerce'));

                    self::send_failed_order_email($order_id);

                    // Remove cart
                    WC()->cart->empty_cart();
                }
            }
		}

        public function send_new_order_email( $order_id ) {
            $emails = WC()->mailer()->get_emails();
            if ( ! empty( $emails ) && ! empty( $order_id ) ) {
                $emails['WC_Email_New_Order']->trigger( $order_id );
                $emails['WC_Email_Customer_Processing_Order']->trigger( $order_id );
            }
        }

        public function send_failed_order_email( $order_id ) {
            $emails = WC()->mailer()->get_emails();
            if ( ! empty( $emails ) && ! empty( $order_id ) ) {
                $emails['WC_Email_Failed_Order']->trigger( $order_id );
            }
        }



		/**
		 * Allow this class and other classes to add slug keyed notices (to avoid duplication)
		 */
		public function add_admin_notice( $slug, $class, $message ) {
			$this->notices[ $slug ] = array(
				'class'   => $class,
				'message' => $message,
			);
		}

		/**
		 * The backup sanity check, in case the plugin is activated in a weird way,
		 * or the environment changes after activation. Also handles upgrade routines.
		 */
		public function check_environment() {
			if ( ! defined( 'IFRAME_REQUEST' ) && ( WC_Worldnet_VERSION !== get_option( 'wc_worldnet_version' ) ) ) {
				$this->install();

				do_action( 'woocommerce_worldnet_updated' );
			}

			$environment_warning = self::get_environment_warning();

			if ( $environment_warning && is_plugin_active( plugin_basename( __FILE__ ) ) ) {
				$this->add_admin_notice( 'bad_environment', 'error', $environment_warning );
			}

			if ( empty( $secret ) && ! ( isset( $_GET['page'], $_GET['section'] ) && 'wc-settings' === $_GET['page'] && 'worldnet' === $_GET['section'] ) ) {
				$setting_link = $this->get_setting_link();
				//$this->add_admin_notice( 'prompt_connect', 'notice notice-warning', sprintf( __( 'Worldnet is almost ready. To get started, <a href="%s">set your Worldnet account keys</a>.', 'woocommerce-gateway-worldnet' ), $setting_link ) );
			}
		}

		/**
		 * Updates the plugin version in db
		 *
		 * @since 3.1.0
		 * @version 3.1.0
		 * @return bool
		 */
		private static function _update_plugin_version() {
			delete_option( 'wc_worldnet_version' );
			update_option( 'wc_worldnet_version', WC_Worldnet_VERSION );

			return true;
		}

		/**
		 * Dismiss the Google Payment Request API Feature notice.
		 *
		 * @since 3.1.0
		 * @version 3.1.0
		 */
		public function dismiss_request_api_notice() {
			update_option( 'wc_worldnet_show_request_api_notice', 'no' );
		}

		/**
		 * Handles upgrade routines.
		 *
		 * @since 3.1.0
		 * @version 3.1.0
		 */
		public function install() {
			if ( ! defined( 'WC_Worldnet_INSTALLING' ) ) {
				define( 'WC_Worldnet_INSTALLING', true );
			}

			$this->_update_plugin_version();
		}

		/**
		 * Checks the environment for compatibility problems.  Returns a string with the first incompatibility
		 * found or false if the environment has no problems.
		 */
		static function get_environment_warning() {
			if ( version_compare( phpversion(), WC_Worldnet_MIN_PHP_VER, '<' ) ) {
				$message = __( 'WooCommerce Worldnet - The minimum PHP version required for this plugin is %1$s. You are running %2$s.', 'woocommerce-gateway-worldnet' );

				return sprintf( $message, WC_Worldnet_MIN_PHP_VER, phpversion() );
			}

			if ( ! defined( 'WC_VERSION' ) ) {
				return __( 'WooCommerce Worldnet requires WooCommerce to be activated to work.', 'woocommerce-gateway-worldnet' );
			}

			if ( version_compare( WC_VERSION, WC_Worldnet_MIN_WC_VER, '<' ) ) {
				$message = __( 'WooCommerce Worldnet - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', 'woocommerce-gateway-worldnet' );

				return sprintf( $message, WC_Worldnet_MIN_WC_VER, WC_VERSION );
			}

			if ( ! function_exists( 'curl_init' ) ) {
				return __( 'WooCommerce Worldnet - cURL is not installed.', 'woocommerce-gateway-worldnet' );
			}

			return false;
		}

		/**
		 * Adds plugin action links
		 *
		 * @since 1.0.0
		 */
		public function plugin_action_links( $links ) {
			$setting_link = $this->get_setting_link();

			$plugin_links = array(
				/*'<a href="' . $setting_link . '">' . __( 'Settings', 'woocommerce-gateway-worldnet' ) . '</a>',
				'<a href="https://docs.woocommerce.com/document/worldnet/">' . __( 'Docs', 'woocommerce-gateway-worldnet' ) . '</a>',
				'<a href="https://woocommerce.com/contact-us/">' . __( 'Support', 'woocommerce-gateway-worldnet' ) . '</a>',*/
			);
			return array_merge( $plugin_links, $links );
		}

		/**
		 * Get setting link.
		 *
		 * @since 1.0.0
		 *
		 * @return string Setting link
		 */
		public function get_setting_link() {
			$use_id_as_section = function_exists( 'WC' ) ? version_compare( WC()->version, '2.6', '>=' ) : false;

			$section_slug = $use_id_as_section ? 'worldnet' : strtolower( 'WC_Gateway_Worldnet' );

			return admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . $section_slug );
		}

		/**
		 * Display any notices we've collected thus far (e.g. for connection, disconnection)
		 */
		public function admin_notices() {
			$show_request_api_notice = get_option( 'wc_worldnet_show_request_api_notice' );

			if ( empty( $show_request_api_notice ) && false ) {
				// @TODO remove this notice in the future.
				?>
				<div class="notice notice-warning wc-worldnet-request-api-notice is-dismissible"><p><?php esc_html_e( 'New Feature! Worldnet now supports Google Payment Request. Your customers can now use mobile phones with supported browsers such as Chrome to make purchases easier and faster.', 'woocommerce-gateway-worldnet' ); ?></p></div>

				<script type="application/javascript">
					jQuery( '.wc-worldnet-request-api-notice' ).on( 'click', '.notice-dismiss', function() {
						var data = {
							action: 'worldnet_dismiss_request_api_notice'
						};

						jQuery.post( '<?php echo admin_url( 'admin-ajax.php' ); ?>', data );
					});
				</script>

				<?php
			}

			foreach ( (array) $this->notices as $notice_key => $notice ) {
				echo "<div class='" . esc_attr( $notice['class'] ) . "'><p>";
				echo wp_kses( $notice['message'], array( 'a' => array( 'href' => array() ) ) );
				echo '</p></div>';
			}
		}

		/**
		 * Initialize the gateway. Called very early - in the context of the plugins_loaded action
		 *
		 * @since 1.0.0
		 */
		public function init_gateways() {
			if ( class_exists( 'WC_Subscriptions_Order' ) && function_exists( 'wcs_create_renewal_order' ) ) {
				$this->subscription_support_enabled = true;
			}

			if ( class_exists( 'WC_Pre_Orders_Order' ) ) {
				$this->pre_order_enabled = true;
			}

			if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
				return;
			}

			if ( class_exists( 'WC_Payment_Gateway_CC' ) ) {
				include_once( dirname( __FILE__ ) . '/includes/class-wc-gateway-worldnet.php' );
			}

			load_plugin_textdomain( 'woocommerce-gateway-worldnet', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );
			add_filter( 'woocommerce_payment_gateways', array( $this, 'add_gateways' ) );

		}

		/**
		 * Add the gateways to WooCommerce
		 *
		 * @since 1.0.0
		 */
		public function add_gateways( $methods ) {
			$methods[] = 'WC_Gateway_Worldnet';

			return $methods;
		}

		/**
		 * List of currencies supported by Worldnet that has no decimals.
		 *
		 * @return array $currencies
		 */
		public static function no_decimal_currencies() {
			return array(
				'bif', // Burundian Franc
				'djf', // Djiboutian Franc
				'jpy', // Japanese Yen
				'krw', // South Korean Won
				'pyg', // Paraguayan Guaraní
				'vnd', // Vietnamese Đồng
				'xaf', // Central African Cfa Franc
				'xpf', // Cfp Franc
				'clp', // Chilean Peso
				'gnf', // Guinean Franc
				'kmf', // Comorian Franc
				'mga', // Malagasy Ariary
				'rwf', // Rwandan Franc
				'vuv', // Vanuatu Vatu
				'xof', // West African Cfa Franc
			);
		}

		/**
		 * Worldnet uses smallest denomination in currencies such as cents.
		 * We need to format the returned currency from Worldnet into human readable form.
		 *
		 * @param object $balance_transaction
		 * @param string $type Type of number to format
		 */
		public static function format_number( $balance_transaction, $type = 'fee' ) {
			if ( ! is_object( $balance_transaction ) ) {
				return;
			}

			if ( in_array( strtolower( $balance_transaction->currency ), self::no_decimal_currencies() ) ) {
				if ( 'fee' === $type ) {
					return $balance_transaction->fee;
				}

				return $balance_transaction->net;
			}

			if ( 'fee' === $type ) {
				return number_format( $balance_transaction->fee / 100, 2, '.', '' );
			}

			return number_format( $balance_transaction->net / 100, 2, '.', '' );
		}

		/**
		 * Checks Worldnet minimum order value authorized per currency
		 */
		public static function get_minimum_amount() {
			// Check order amount
			switch ( get_woocommerce_currency() ) {
				case 'USD':
				case 'CAD':
				case 'EUR':
				case 'CHF':
				case 'AUD':
				case 'SGD':
					$minimum_amount = 50;
					break;
				case 'GBP':
					$minimum_amount = 30;
					break;
				case 'DKK':
					$minimum_amount = 250;
					break;
				case 'NOK':
				case 'SEK':
					$minimum_amount = 300;
					break;
				case 'JPY':
					$minimum_amount = 5000;
					break;
				case 'MXN':
					$minimum_amount = 1000;
					break;
				case 'HKD':
					$minimum_amount = 400;
					break;
				default:
					$minimum_amount = 50;
					break;
			}

			return $minimum_amount;
		}

		/**
		 * What rolls down stairs
		 * alone or in pairs,
		 * and over your neighbor's dog?
		 * What's great for a snack,
		 * And fits on your back?
		 * It's log, log, log
		 */
		public static function log( $message ) {
			if ( empty( self::$log ) ) {
				self::$log = new WC_Logger();
			}

			self::$log->add( 'woocommerce-gateway-worldnet', $message );
		}
	}

	$GLOBALS['wc_worldnet'] = WC_Worldnet::get_instance();

endif;
