<?php
/**
 * Copyright © 2013-2017 Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Nuvei\Payment\Controller\Directpost\Payment;

use Nuvei\Payment\Controller\Directpost\Payment;
use Nuvei\Payment\Helper\DataFactory;
use Magento\Checkout\Model\Type\Onepage;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Response\Http;
use Magento\Framework\Json\Helper\Data as JsonHelper;
use Magento\Framework\DataObject;
use Magento\Framework\Registry;
use Nuvei\Payment\Model\IframeConfigProvider;
use Nuvei\Payment\Model\Directpost;
use Magento\Quote\Api\CartManagementInterface;
use Magento\Quote\Api\CartRepositoryInterface;

/**
 * Class Place
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Place extends Payment
{
    /**
     * @var \Magento\Quote\Api\CartManagementInterface
     */
    protected $cartManagement;

    /**
     * @var \Magento\Framework\Event\ManagerInterface
     */
    protected $eventManager;

    /**
     * @var \Magento\Checkout\Model\Type\Onepage
     */
    protected $onepageCheckout;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    protected $quoteRepository;

    protected $_logger;

    protected $paymentMethod;

    /**
     * Core store config
     *
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $_scopeConfig;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @param Context $context
     * @param Registry $coreRegistry
     * @param DataFactory $dataFactory
     * @param CartManagementInterface $cartManagement
     * @param Onepage $onepageCheckout
     * @param JsonHelper $jsonHelper
     * @param CartRepositoryInterface $quoteRepository
     * @param Directpost $paymentMethod
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param LoggerInterface $logger
     */
    public function __construct(
        Context $context,
        Registry $coreRegistry,
        DataFactory $dataFactory,
        CartManagementInterface $cartManagement,
        Onepage $onepageCheckout,
        JsonHelper $jsonHelper,
        CartRepositoryInterface $quoteRepository,
        Directpost $paymentMethod,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->eventManager = $context->getEventManager();
        $this->cartManagement = $cartManagement;
        $this->onepageCheckout = $onepageCheckout;
        $this->jsonHelper = $jsonHelper;
        $this->quoteRepository = $quoteRepository;
        $this->paymentMethod = $paymentMethod;
        $this->_scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->_logger = $logger;
        parent::__construct($context, $coreRegistry, $dataFactory);
    }



    /**
     * Retrieve information from payment configuration
     *
     * @param string $field
     * @param int|string|null|\Magento\Store\Model\Store $storeId
     *
     * @return mixed
     */
    public function getConfigData($field)
    {
        $path = 'payment/nuvei_directpost/' . $field;
        return $this->_scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITES);
    }

    public function getField($field) {
        if ($this->getConfigData('mode') == 'LIVE')
            return  ($this->getConfigData($field));
        else
            return  ($this->getConfigData('test_'.$field));
    }

    /**
     * Send request to Nuvei
     *
     * @return string
     */
    public function execute()
    {
        $paymentParam = $this->getRequest()->getParam('payment');
        $controller = $this->getRequest()->getParam('controller');
        $response = $this->getResponse();

        if (isset($paymentParam['method'])) {
            //$this->_getDirectPostSession()->setQuoteId($this->_getCheckout()->getQuote()->getId());
            /**
             * Current workaround depends on Onepage checkout model defect
             * Method Onepage::getCheckoutMethod performs setCheckoutMethod
             */
            $this->onepageCheckout->getCheckoutMethod();

            if ($controller == IframeConfigProvider::CHECKOUT_IDENTIFIER) {
                return $this->placeCheckoutOrder();
            }

            $params = $this->dataFactory
                ->create(DataFactory::AREA_FRONTEND)
                ->getSaveOrderUrlParams($controller);
            $this->_forward(
                $params['action'],
                $params['controller'],
                $params['module'],
                $this->getRequest()->getParams()
            );
        } else {
            $result = ['error_messages' => __('Please choose a payment method.'), 'goto_section' => 'payment'];
            if ($response instanceof Http) {
                $response->representJson($this->jsonHelper->jsonEncode($result));
            }
        }
    }

    /**
     * Place order for checkout flow
     *
     * @return string
     */
    protected function placeCheckoutOrder()
    {
        $result = new DataObject();
        $response = $this->getResponse();
        try {
            //$this->cartManagement->placeOrder($this->_getCheckout()->getQuote()->getId());
            $quoteId = $this->_getCheckout()->getQuote()->getId();
            $quote = $this->quoteRepository->getActive($quoteId);
            $address = $quote->getBillingAddress();
            $street = $address->getStreet();
            $customer = $quote->getCustomer();

            $fields = [];
            $fields['controller_action_name'] = 'directpost_payment';
            $fields['is_secure'] = (string)$this->storeManager->getStore()->isCurrentlySecure();
            $fields['x_quote_id'] = $quoteId;
            $fields['x_address'] = isset($street[0])?$street[0]:'';
            $fields['x_address_2'] = isset($street[1])?$street[1]:'';
            $fields['x_amount'] = $quote->getBaseGrandTotal();
            $fields['x_city'] = $address->getCity();
            $fields['x_company'] = $address->getCompany();
            $fields['x_country'] = $address->getCountryId();
            $fields['x_currency_code'] = $quote->getQuoteCurrencyCode();
            $fields['x_cust_id'] = $customer->getId();
            $fields['x_customer_ip'] = $quote->getRemoteIp();
            $fields['x_customer_tax_id'] = $customer->getTaxvat();
            $fields['x_delim_data'] = 'FALSE';
            $fields['x_email'] = $customer->getEmail();
            $fields['x_email_customer'] = $this->getConfigData('email_customer');
            $fields['x_fax'] = '';
            $fields['x_first_name'] = $quote->getCustomerFirstname();
            $fields['x_fp_sequence'] = $quoteId;
            $fields['x_freight'] = $quote->getBaseShippingAmount();
            $fields['x_invoice_num'] = 'quote_'.$quoteId;
            $fields['x_last_name'] = $quote->getCustomerLastname();
            $fields['x_merchant_email'] = $this->getConfigData('merchant_email');
            $fields['x_method'] = 'CC';
            $fields['x_phone'] = $address->getTelephone();
            $fields['x_po_num'] = '';
            $fields['x_region'] = $address->getRegion();
            $fields['x_relay_response'] = 'TRUE';
            $fields['x_relay_url'] = $this->paymentMethod->getRelayUrl();
            $fields['x_state'] = $address->getRegion();
            $fields['x_tax'] = $quote->getBaseTaxAmount();
            $fields['x_test_request'] = $this->getConfigData('mode')=='TEST' ? 'TRUE' : 'FALSE';
            $fields['x_type'] = $this->getConfigData('payment_action');
            $fields['x_version'] = '3.1';
            $fields['x_zip'] = $address->getPostcode();

            if($fields['x_type'] == $this->paymentMethod::ACTION_AUTHORIZE) {
                $fields['x_type'] = $this->paymentMethod::REQUEST_TYPE_AUTH_ONLY;
            } else if($fields['x_type'] == $this->paymentMethod::ACTION_AUTHORIZE_CAPTURE) {
                $fields['x_type'] = $this->paymentMethod::REQUEST_TYPE_AUTH_CAPTURE;
            }

            $nuveiDirectpost = ['fields' => $fields];

            $result->setData('nuvei_directpost', $nuveiDirectpost);
            $result->setData('success', true);

            $this->eventManager->dispatch(
                'checkout_directpost_placeOrder',
                [
                    'result' => $result,
                    'action' => $this
                ]
            );
        } catch (\Exception $exception) {
            $result->setData('error', true);
            $result->setData(
                'error_messages',
                __('An error occurred on the server. Please try to place the order again.')
            );
        }
        if ($response instanceof Http) {
            $response->representJson($this->jsonHelper->jsonEncode($result));
        }
    }
}
