<?php
/*
 * Plugin Name: Nuvei Gateway for WooCommerce
 * Description: Take credit card payments on your store using Nuvei.
 * Author: Nuvei
 * Author URI: https://www.nuvei.com/
 * Version: 2.5.2
 * Text Domain: nuvei-gateway-for-woocommerce
 *
 * Tested up to: 5.4.2
 * WC tested up to: 4.2.0
 *
 * Copyright (c) 2020 Nuvei
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Required minimums and constants
 */
define( 'Nuvei_Gateway_for_WC_VERSION', '2.5.2' );
define( 'Nuvei_Gateway_for_WC_MIN_PHP_VER', '5.6.0' );
define( 'Nuvei_Gateway_for_WC_MIN_WC_VER', '2.5.0' );
define( 'Nuvei_Gateway_for_WC_MAIN_FILE', __FILE__ );
define( 'WC_Nuvei_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );

class Nuvei_Gateway_for_WC {

    /**
     * @var Singleton The reference the *Singleton* instance of this class
     */
    private static $instance;

    /**
     * @var Reference to logging class.
     */
    private static $log;

    /**
     * Returns the *Singleton* instance of this class.
     *
     * @return Singleton The *Singleton* instance.
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private clone method to prevent cloning of the instance of the
     * *Singleton* instance.
     *
     * @return void
     */
    private function __clone() {}

    /**
     * Private unserialize method to prevent unserializing of the *Singleton*
     * instance.
     *
     * @return void
     */
    private function __wakeup() {}

    /**
     * Flag to indicate whether or not we need to load code for / support subscriptions.
     *
     * @var bool
     */
    private $subscription_support_enabled = false;

    /**
     * Flag to indicate whether or not we need to load support for pre-orders.
     *
     * @since 3.0.3
     *
     * @var bool
     */
    private $pre_order_enabled = false;

    /**
     * Notices (array)
     * @var array
     */
    public $notices = array();

    /**
     * Protected constructor to prevent creating a new instance of the
     * *Singleton* via the `new` operator from outside of this class.
     */
    protected function __construct() {
        add_action( 'admin_init', array( $this, 'check_environment' ) );
        add_action( 'admin_notices', array( $this, 'admin_notices' ), 15 );
        add_action( 'plugins_loaded', array( $this, 'init' ) );
    }

    /**
     * Init the plugin after plugins_loaded so environment variables are set.
     */
    public function init() {
        // Don't hook anything else in the plugin if we're in an incompatible environment
        if ( self::get_environment_warning() ) {
            return;
        }

        // Init the gateway itself
        $this->init_gateways();

        // Background Validation
        $options = get_option("woocommerce_nuvei_settings");
        $background_validation = isset($options['background_validation'])?$options['background_validation']:false;

        function runNuveiOnInit() {
            $uniqueref = wc_clean(wp_unslash($_POST['UNIQUEREF']));
            $order_id = wc_clean(wp_unslash($_POST['ORDERID']));
            $responseCode = wc_clean(wp_unslash($_POST['RESPONSECODE']));

            $order = wc_get_order($order_id);
            $orderStatus = $order->get_status();
            $orderStatus = 'wc-' === substr( $orderStatus, 0, 3 ) ? substr( $orderStatus, 3 ) : $orderStatus;

            if($responseCode == 'A') {
                if($orderStatus == 'pending' || $orderStatus == 'failed') {
                    $order->update_status( apply_filters( 'woocommerce_gateway_nuvei_process_payment_order_status', 'processing', $order ), __( 'Payment processed correctly and Background Validation completed #'. $uniqueref, 'woocommerce' ) );

                    $order->set_transaction_id( $uniqueref );

                    $order->save();
                } else {
                    $order->add_order_note( __( 'Background Validation completed #'. $uniqueref, 'woocommerce' ) );
                }
            } else {
                if($orderStatus == 'pending') {
                    $order->update_status(apply_filters('woocommerce_gateway_nuvei_process_payment_order_status', 'failed', $order), __('Payment failed #' . $uniqueref, 'woocommerce'));
                } else {
                    $order->add_order_note( __( 'Background Validation completed #'. $uniqueref, 'woocommerce' ) );
                }
            }

            echo 'OK';
            exit;
        }
        if(isset($_POST['UNIQUEREF'])) {
            if ( $background_validation == "yes" ) {
                $terminalId = wc_clean(wp_unslash($_POST['TERMINALID']));
                $order_id = wc_clean(wp_unslash($_POST['ORDERID']));
                $amount = wc_clean(wp_unslash($_POST['AMOUNT']));
                $dateTime = wc_clean(wp_unslash($_POST['DATETIME']));
                $responseCode = wc_clean(wp_unslash($_POST['RESPONSECODE']));
                $responseText = wc_clean(wp_unslash($_POST['RESPONSETEXT']));
                $hash = wc_clean(wp_unslash($_POST['HASH']));

                $env = "live_";
                if($options['testmode'] == "yes")
                    $env = "test_";

                if($terminalId == $options[$env.'publishable_key2']) {
                    $secret = $options[$env.'secret_key2'];
                    $currency = $options[$env.'currency2'];
                    $multicurrency = $options[$env.'multicurrency2'];
                } else if($terminalId == $options[$env.'publishable_key3']) {
                    $secret = $options[$env.'secret_key3'];
                    $currency = $options[$env.'currency3'];
                    $multicurrency = $options[$env.'multicurrency3'];
                } else {
                    $secret = $options[$env.'secret_key'];
                    $currency = $options[$env.'currency'];
                    $multicurrency = $options[$env.'multicurrency'];
                }

                $expectedHash = md5($terminalId . $order_id . ($multicurrency=='yes' ? $currency : '') . $amount . $dateTime . $responseCode . $responseText . $secret);

                if($expectedHash == $hash)
                    add_action( 'init', 'runNuveiOnInit' );
                else {
                    echo 'Incorrect HASH';
                    exit;
                }
            } else {
                    echo 'Background Validation not enabled in Nuvei plugin';
                    exit;
            }
        }
        // ----------------- End of Background Validation

        add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );
        add_action( 'wp_ajax_nuvei_dismiss_request_api_notice', array( $this, 'dismiss_request_api_notice' ) );

        add_action( 'woocommerce_thankyou', array( $this, 'checkResponse'), 10, 1 );
        add_action( 'woocommerce_before_checkout_form', array( $this, 'check3DSResponse'), 10, 1 );
    }

    public function check3DSResponse() {
        if(isset($_GET['ORDERID'])) {
            $order_id = absint($_GET['ORDERID']);

            if(isset($_GET['RESULT']) && sanitize_text_field($_GET['RESULT']) == 'D') {
                $order = wc_get_order($order_id);
                $orderStatus = $order->get_status();
                $orderStatus = 'wc-' === substr( $orderStatus, 0, 3 ) ? substr( $orderStatus, 3 ) : $orderStatus;

                $receiptPageUrl = $order->get_checkout_order_received_url();

                if($orderStatus != 'failed') {
                    $order->update_status(apply_filters('woocommerce_gateway_nuvei_process_payment_order_status', 'failed', $order), __('3DS failed; STATUS='.sanitize_text_field($_GET['STATUS']).'; ECI='.sanitize_text_field($_GET['ECI']), 'woocommerce'));

                    // Remove cart
                    WC()->cart->empty_cart();
                    WC()->session->__unset( 'wc_gateway_nuvei_'.$order_id );
                }

                wp_redirect($receiptPageUrl);
            } else if(isset($_GET['RESULT']) && sanitize_text_field($_GET['RESULT']) == 'A') {
                $cardDetails = WC()->session->get( 'wc_gateway_nuvei_'.$order_id);
                $order = wc_get_order($order_id);

                $testmode = $cardDetails['testmode'];
                $avs = $cardDetails['avs'];
                $multicurrency = $cardDetails['multicurrency'];
                $terminalId = $cardDetails['terminalId'];
                $secret = $cardDetails['secret'];
                $currency = $cardDetails['currency'];
                $amount = $cardDetails['amount'];
                $cardNumber = $cardDetails['cardNumber'];
                $cardHolder = $cardDetails['cardHolder'];
                $cardExpiry = $cardDetails['cardExpiry'];
                $cardType = $cardDetails['cardType'];
                $cvv = $cardDetails['cvv'];

                $XmlAuthRequest = new NuveiGatewayXmlAuthRequest($terminalId, $order_id, $currency, $amount, $cardNumber, $cardType);

                $XmlAuthRequest->SetTransactionType("5");

                if ($cardType != "SECURECARD") $XmlAuthRequest->SetNonSecureCardCardInfo($cardExpiry, $cardHolder);
                if ($cvv != "") $XmlAuthRequest->SetCvv($cvv);
                if ($multicurrency=='yes') $XmlAuthRequest->SetMultiCur();
                if (isset($_GET['MPIREF'])) $XmlAuthRequest->SetMpiRef(wc_clean(wp_unslash($_GET['MPIREF'])));
                if (isset($_GET['XID'])) $XmlAuthRequest->SetXid(wc_clean(wp_unslash($_GET['XID'])));
                if (isset($_GET['CAVV'])) $XmlAuthRequest->SetCavv(wc_clean(wp_unslash($_GET['CAVV'])));


                if(true || $avs) { //always send AVS data on xml requests and let gateway decide if will use it or not
                    $address1 = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_1 : $order->get_billing_address_1();
                    $address2 = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_2 : $order->get_billing_address_2();
                    $postcode = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_postcode : $order->get_billing_postcode();
                    $city = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_city : $order->get_billing_city();
                    $region = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_state : $order->get_billing_state();
                    $country = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_country : $order->get_billing_country();
                    $email = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_email : $order->get_billing_email();
                    $phone = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_phone : $order->get_billing_phone();


                    $XmlAuthRequest->SetAvs($address1, $address2, $postcode);
                    $XmlAuthRequest->SetCity($city);
                    if($region != "") $XmlAuthRequest->SetRegion($region);
                    $XmlAuthRequest->SetCountry($country);
                    $XmlAuthRequest->SetEmail($email);
                    $XmlAuthRequest->SetPhone($phone);
                }

                # Perform the online authorisation and read in the result
                $serverUrl = $testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
                $response = $XmlAuthRequest->ProcessRequestToGateway($secret, $serverUrl);

                $expectedResponseHash = md5($terminalId . $response->UniqueRef() . ($multicurrency=='yes' ? $currency : '')  . $amount . $response->DateTime() . $response->ResponseCode() . $response->ResponseText() . $secret);
                $isHashCorrect = ($expectedResponseHash == $response->Hash());

                $nuveiResponse = '';

                if ($response->IsError()) $nuveiResponse .= 'AN ERROR OCCURED! Your transaction was not processed. Error details: ' . $response->ErrorString();
                elseif ($isHashCorrect) {
                    switch ($response->ResponseCode()) {
                        case "A" :    # -- If using local database, update order as Authorised.
                            $nuveiResponse .= 'Payment Processed successfully. Thanks you for your order.';
                            $uniqueRef = $response->UniqueRef();
                            $responseText = $response->ResponseText();
                            $approvalCode = $response->ApprovalCode();
                            $avsResponse = $response->AvsResponse();
                            $cvvResponse = $response->CvvResponse();
                            break;
                        case "R" :
                        case "D" :
                        case "C" :
                        case "S" :
                        default  :    # -- If using local database, update order as declined/failed --
                            $nuveiResponse .= 'PAYMENT DECLINED! Please try again with another card. Bank response: ' . $response->ResponseText();
                    }
                } else {
                    $nuveiResponse .= 'PAYMENT FAILED: INVALID RESPONSE HASH. Please contact ' . $adminEmail . ' or call ' . $adminPhone . ' to clarify if you will get charged for this order.';
                    if ($response->UniqueRef()) $nuveiResponse .= 'Please quote Nuvei Terminal ID: ' . $terminalId . ', and Unique Reference: ' . $response->UniqueRef() . ' when mailing or calling.';
                }

                $receiptPageUrl = $order->get_checkout_order_received_url();

                if (!$response->IsError() && $isHashCorrect && $response->ResponseCode() == 'A') {
                    $order->update_status( apply_filters( 'woocommerce_gateway_nuvei_process_payment_order_status', 'processing', $order ), __( 'Payment successfully processed #'. $response->UniqueRef(), 'woocommerce' ) );

                    $order->set_transaction_id( $response->UniqueRef() );

                    $order->save();

                    WC()->mailer()->customer_invoice( $order );

                    wc_reduce_stock_levels( $order_id );

                    WC()->cart->empty_cart();

                    wp_redirect($receiptPageUrl);
                }
                else {
                    $order->update_status(apply_filters('woocommerce_gateway_nuvei_process_payment_order_status', 'failed', $order), __('Payment failed: '.$nuveiResponse, 'woocommerce'));

                    // Remove cart
                    WC()->cart->empty_cart();
                    WC()->session->__unset( 'wc_gateway_nuvei_'.$order_id );

                    wp_redirect($receiptPageUrl);
                }
            }
        }
    }

    public function checkResponse() {
        if (isset($_GET['ORDERID']) && wc_clean(wp_unslash($_GET['ORDERID'])) && isset($_GET['HASH']) && wc_clean(wp_unslash($_GET['HASH']))) {
            if (sanitize_text_field($_GET['RESPONSECODE']) == 'A') { //TODO: perform hash background validation
                // Lets grab the order
                $order_id = wc_get_order_id_by_order_key(urldecode($_GET['key']));
                $order = wc_get_order($order_id);
                $orderStatus = $order->get_status();
                $orderStatus = 'wc-' === substr( $orderStatus, 0, 3 ) ? substr( $orderStatus, 3 ) : $orderStatus;

                if($orderStatus == 'pending' || $orderStatus == 'failed') {
                    $order->update_status(apply_filters('woocommerce_order_status_pending_to_processing_notification', 'processing', $order), __('Payment processed correctly #' . sanitize_text_field($_GET['UNIQUEREF']), 'woocommerce'));

                    $order->set_transaction_id( sanitize_text_field($_GET['UNIQUEREF']) );

                    $order->save();

                    WC()->mailer()->customer_invoice($order);

                    wc_reduce_stock_levels( $order_id );

                    // Remove cart
                    WC()->cart->empty_cart();

                    // Redirect merchant to receipt page
                    $receiptPageUrl = $order->get_checkout_order_received_url();
                    wp_redirect($receiptPageUrl);
                }
            } else {
                $currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://{$_SERVER['HTTP_HOST']}{$_SERVER['REQUEST_URI']}";

                $order_id = wc_get_order_id_by_order_key(urldecode($_GET['key']));
                $order = wc_get_order($order_id);
                $orderStatus = $order->get_status();
                $orderStatus = 'wc-' === substr( $orderStatus, 0, 3 ) ? substr( $orderStatus, 3 ) : $orderStatus;


                if($orderStatus == 'pending') {
                    $order->update_status(apply_filters('woocommerce_gateway_nuvei_process_payment_order_status', 'failed', $order), __('Payment failed #' . sanitize_text_field($_GET['UNIQUEREF']), 'woocommerce'));

                    // Remove cart
                    WC()->cart->empty_cart();

                    wp_redirect($currentUrl);
                }
            }
        }
    }

    public function send_new_order_email( $order_id ) {
        $emails = WC()->mailer()->get_emails();
        if ( ! empty( $emails ) && ! empty( $order_id ) ) {
            $emails['WC_Email_New_Order']->trigger( $order_id );
            $emails['WC_Email_Customer_Processing_Order']->trigger( $order_id );
        }
    }

    public function send_failed_order_email( $order_id ) {
        $emails = WC()->mailer()->get_emails();
        if ( ! empty( $emails ) && ! empty( $order_id ) ) {
            $emails['WC_Email_Failed_Order']->trigger( $order_id );
        }
    }



    /**
     * Allow this class and other classes to add slug keyed notices (to avoid duplication)
     */
    public function add_admin_notice( $slug, $class, $message ) {
        $this->notices[ $slug ] = array(
            'class'   => $class,
            'message' => $message,
        );
    }

    /**
     * The backup sanity check, in case the plugin is activated in a weird way,
     * or the environment changes after activation. Also handles upgrade routines.
     */
    public function check_environment() {
        if ( ! defined( 'IFRAME_REQUEST' ) && ( Nuvei_Gateway_for_WC_VERSION !== get_option( 'wc_nuvei_version' ) ) ) {
            $this->install();

            do_action( 'woocommerce_nuvei_updated' );
        }

        $environment_warning = self::get_environment_warning();

        if ( $environment_warning && is_plugin_active( plugin_basename( __FILE__ ) ) ) {
            $this->add_admin_notice( 'bad_environment', 'error', $environment_warning );
        }
    }

    /**
     * Updates the plugin version in db
     *
     * @since 3.1.0
     * @version 3.1.0
     * @return bool
     */
    private static function _update_plugin_version() {
        delete_option( 'wc_nuvei_version' );
        update_option( 'wc_nuvei_version', Nuvei_Gateway_for_WC_VERSION );

        return true;
    }

    /**
     * Dismiss the Google Payment Request API Feature notice.
     *
     * @since 3.1.0
     * @version 3.1.0
     */
    public function dismiss_request_api_notice() {
        update_option( 'wc_nuvei_show_request_api_notice', 'no' );
    }

    /**
     * Handles upgrade routines.
     *
     * @since 3.1.0
     * @version 3.1.0
     */
    public function install() {
        if ( ! defined( 'Nuvei_Gateway_for_WC_INSTALLING' ) ) {
            define( 'Nuvei_Gateway_for_WC_INSTALLING', true );
        }

        $this->_update_plugin_version();
    }

    /**
     * Checks the environment for compatibility problems.  Returns a string with the first incompatibility
     * found or false if the environment has no problems.
     */
    static function get_environment_warning() {
        if ( version_compare( phpversion(), Nuvei_Gateway_for_WC_MIN_PHP_VER, '<' ) ) {
            $message = __( 'WooCommerce Nuvei - The minimum PHP version required for this plugin is %1$s. You are running %2$s.', 'woocommerce-gateway-nuvei' );

            return sprintf( $message, Nuvei_Gateway_for_WC_MIN_PHP_VER, phpversion() );
        }

        if ( ! defined( 'WC_VERSION' ) ) {
            return __( 'WooCommerce Nuvei requires WooCommerce to be activated to work.', 'woocommerce-gateway-nuvei' );
        }

        if ( version_compare( WC_VERSION, Nuvei_Gateway_for_WC_MIN_WC_VER, '<' ) ) {
            $message = __( 'WooCommerce Nuvei - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', 'woocommerce-gateway-nuvei' );

            return sprintf( $message, Nuvei_Gateway_for_WC_MIN_WC_VER, WC_VERSION );
        }

        if ( ! function_exists( 'curl_init' ) ) {
            return __( 'WooCommerce Nuvei - cURL is not installed.', 'woocommerce-gateway-nuvei' );
        }

        return false;
    }

    /**
     * Adds plugin action links
     *
     * @since 1.0.0
     */
    public function plugin_action_links( $links ) {
        $setting_link = $this->get_setting_link();

        $plugin_links = array(
            '<a href="' . $setting_link . '">' . __( 'Settings', 'woocommerce-gateway-nuvei' ) . '</a>',
            '<a href="mailto:support@nuvei.com">' . __( 'Support', 'woocommerce-gateway-nuvei' ) . '</a>',
        );
        return array_merge( $plugin_links, $links );
    }

    /**
     * Get setting link.
     *
     * @since 1.0.0
     *
     * @return string Setting link
     */
    public function get_setting_link() {
        $use_id_as_section = function_exists( 'WC' ) ? version_compare( WC()->version, '2.6', '>=' ) : false;

        $section_slug = $use_id_as_section ? 'nuvei' : strtolower( 'Nuvei_Gateway' );

        return admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . $section_slug );
    }

    /**
     * Display any notices we've collected thus far (e.g. for connection, disconnection)
     */
    public function admin_notices() {
        $show_request_api_notice = get_option( 'wc_nuvei_show_request_api_notice' );

        if ( empty( $show_request_api_notice ) && false ) {
            // @TODO remove this notice in the future.
            ?>
            <div class="notice notice-warning wc-nuvei-request-api-notice is-dismissible"><p><?php esc_html_e( 'New Feature! Nuvei now supports Google Payment Request. Your customers can now use mobile phones with supported browsers such as Chrome to make purchases easier and faster.', 'woocommerce-gateway-nuvei' ); ?></p></div>

            <script type="application/javascript">
                jQuery( '.wc-nuvei-request-api-notice' ).on( 'click', '.notice-dismiss', function() {
                    var data = {
                        action: 'nuvei_dismiss_request_api_notice'
                    };

                    jQuery.post( '<?php echo admin_url( 'admin-ajax.php' ); ?>', data );
                });
            </script>

            <?php
        }

        foreach ( (array) $this->notices as $notice_key => $notice ) {
            echo "<div class='" . esc_attr( $notice['class'] ) . "'><p>";
            echo wp_kses( $notice['message'], array( 'a' => array( 'href' => array() ) ) );
            echo '</p></div>';
        }
    }

    /**
     * Initialize the gateway. Called very early - in the context of the plugins_loaded action
     *
     * @since 1.0.0
     */
    public function init_gateways() {
        if ( class_exists( 'WC_Subscriptions_Order' ) && function_exists( 'wcs_create_renewal_order' ) ) {
            $this->subscription_support_enabled = true;
        }

        if ( class_exists( 'WC_Pre_Orders_Order' ) ) {
            $this->pre_order_enabled = true;
        }

        if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
            return;
        }

        if ( class_exists( 'WC_Payment_Gateway_CC' ) ) {
            include_once( dirname( __FILE__ ) . '/includes/class-nuvei-gateway.php' );
        }

        load_plugin_textdomain( 'woocommerce-gateway-nuvei', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );
        add_filter( 'woocommerce_payment_gateways', array( $this, 'add_gateways' ) );

    }

    /**
     * Add the gateways to WooCommerce
     *
     * @since 1.0.0
     */
    public function add_gateways( $methods ) {
        $methods[] = 'Nuvei_Gateway';

        return $methods;
    }

    /**
     * List of currencies supported by Nuvei that has no decimals.
     *
     * @return array $currencies
     */
    public static function no_decimal_currencies() {
        return array(
            'bif', // Burundian Franc
            'djf', // Djiboutian Franc
            'jpy', // Japanese Yen
            'krw', // South Korean Won
            'pyg', // Paraguayan Guaraní
            'vnd', // Vietnamese Đồng
            'xaf', // Central African Cfa Franc
            'xpf', // Cfp Franc
            'clp', // Chilean Peso
            'gnf', // Guinean Franc
            'kmf', // Comorian Franc
            'mga', // Malagasy Ariary
            'rwf', // Rwandan Franc
            'vuv', // Vanuatu Vatu
            'xof', // West African Cfa Franc
        );
    }

    /**
     * Nuvei uses smallest denomination in currencies such as cents.
     * We need to format the returned currency from Nuvei into human readable form.
     *
     * @param object $balance_transaction
     * @param string $type Type of number to format
     */
    public static function format_number( $balance_transaction, $type = 'fee' ) {
        if ( ! is_object( $balance_transaction ) ) {
            return;
        }

        if ( in_array( strtolower( $balance_transaction->currency ), self::no_decimal_currencies() ) ) {
            if ( 'fee' === $type ) {
                return $balance_transaction->fee;
            }

            return $balance_transaction->net;
        }

        if ( 'fee' === $type ) {
            return number_format( $balance_transaction->fee / 100, 2, '.', '' );
        }

        return number_format( $balance_transaction->net / 100, 2, '.', '' );
    }

    public static function log( $message ) {
        if ( empty( self::$log ) ) {
            self::$log = new WC_Logger();
        }

        self::$log->add( 'woocommerce-gateway-nuvei', $message );
    }
}

$GLOBALS['wc_nuvei'] = Nuvei_Gateway_for_WC::get_instance();
