<?php
/*
    Copyright 2020 Nuvei

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

if(basename(__FILE__) == basename($_SERVER["REQUEST_URI"])) die("<b>ERROR: You cannot display this file directly.</b>");

/**
 * Base NuveiGatewayRequest Class holding common functionality for Request Types.
 */
class NuveiGatewayRequest
{
		protected static function GetRequestHash($plainString)
		{
				return md5($plainString);
		}

		protected static function GetFormattedDate()
		{
				return date('d-m-Y:H:i:s:000');
		}

		protected static function SendRequestToGateway($requestString, $serverUrl, $headers = array())
		{
                $args = array(
                    'body' => $requestString,
                    'timeout' => '5',
                    'redirection' => '5',
                    'httpversion' => '1.0',
                    'blocking' => true,
                    'headers' => $headers,
                    'cookies' => array()
                );
                $response = wp_remote_post( $serverUrl, $args );

                return $response['body'];
		}
}

/**
 *  Used for processing XML Authorisations through the Nuvei XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class NuveiGatewayXmlAuthRequest extends NuveiGatewayRequest
{
		private $terminalId;
		private $orderId;
		private $currency;
		private $amount;
		public function Amount()
		{
				return $this->amount;
		}
		private $dateTime;
		private $hash;
		private $autoReady;
		private $description;
		private $email;
		private $cardNumber;
		private $trackData;
		private $cardType;
		private $cardExpiry;
		private $cardHolderName;
		private $cvv;
		private $issueNo;
		private $address1;
		private $address2;
		private $postCode;
		private $cardCurrency;
		private $cardAmount;
		private $conversionRate;
		private $terminalType = "2";
		private $transactionType = "7";
		private $avsOnly;
        private $mpiRef;
        private $xid;
        private $cavv;
		private $mobileNumber;
		private $deviceId;
		private $phone;
        private $city;
        private $region;
        private $country;
		private $ipAddress;

		private $multicur = false;
		private $foreignCurInfoSet = false;

		/**
		 *  Creates the standard request less optional parameters for processing an XML Transaction
		 *  through the Nuvei XML Gateway
		 *
		 *  @param terminalId Terminal ID provided by Nuvei
		 *  @param orderId A unique merchant identifier. Alpha numeric and max size 12 chars.
		 *  @param currency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param amount Transaction Amount, Double formatted to 2 decimal places.
		 *  @param description Transaction Description
		 *  @param email Cardholder e-mail
		 *  @param cardNumber A valid Card Number that passes the Luhn Check.
		 *  @param cardType
		 *  Card Type (Accepted Card Types must be configured in the Merchant Selfcare System.)
		 *
		 *  Accepted Values :
		 *
		 *  VISA
		 *  MASTERCARD
		 *  LASER
		 *  SWITCH
		 *  SOLO
		 *  AMEX
		 *  DINERS
		 *  MAESTRO
		 *  DELTA
		 *  ELECTRON
		 *
		 */
		public function __construct($terminalId,
				$orderId,
				$currency,
				$amount,
				$cardNumber,
				$cardType)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->terminalId = $terminalId;
				$this->orderId = $orderId;
				$this->currency = $currency;
				$this->amount = $amount;
				$this->cardNumber = $cardNumber;
				$this->cardType = $cardType;
		}
	   /**
		 *  Setter for Auto Ready Value
		 *
		 *  @param autoReady
		 *  Auto Ready is an optional parameter and defines if the transaction should be settled automatically.
		 *
		 *  Accepted Values :
		 *
		 *  Y   -   Transaction will be settled in next batch
		 *  N   -   Transaction will not be settled until user changes state in Merchant Selfcare Section
		 */
		public function SetAutoReady($autoReady)
		{
				$this->autoReady = $autoReady;
		}
	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetEmail($email)
		{
				$this->email = $email;
		}
	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetDescription($description)
		{
				$this->description = $description;
		}
	   /**
		 *  Setter for Card Expiry and Card Holder Name values
		 *  These are mandatory for non-SecureCard transactions
		 *
		 *  @param cardExpiry Card Expiry formatted MMYY
		 *  @param cardHolderName Card Holder Name
		 */
		public function SetNonSecureCardCardInfo($cardExpiry, $cardHolderName)
		{
				$this->cardExpiry = $cardExpiry;
				$this->cardHolderName = $cardHolderName;
		}
	   /**
		 *  Setter for Card Verification Value
		 *
		 *  @param cvv Numeric field with a max of 4 characters.
		 */
		public function SetCvv($cvv)
		{
				$this->cvv = $cvv;
		}

	   /**
		 *  Setter for Issue No
		 *
		 *  @param issueNo Numeric field with a max of 3 characters.
		 */
		public function SetIssueNo($issueNo)
		{
				$this->issueNo = $issueNo;
		}

	   /**
		 *  Setter for Address Verification Values
		 *
		 *  @param address1 First Line of address - Max size 20
		 *  @param address2 Second Line of address - Max size 20
		 *  @param postCode Postcode - Max size 9
		 */
		public function SetAvs($address1, $address2, $postCode)
		{
				$this->address1 = $address1;
				$this->address2 = $address2;
				$this->postCode = $postCode;
		}
	   /**
		 *  Setter for Foreign Currency Information
		 *
		 *  @param cardCurrency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param cardAmount (Amount X Conversion rate) Formatted to two decimal places
		 *  @param conversionRate Converstion rate supplied in rate response
		 */
		public function SetForeignCurrencyInformation($cardCurrency, $cardAmount, $conversionRate)
		{
				$this->cardCurrency = $cardCurrency;
				$this->cardAmount = $cardAmount;
				$this->conversionRate = $conversionRate;

				$this->foreignCurInfoSet = true;
		}
	   /**
		 *  Setter for AVS only flag
		 *
		 *  @param avsOnly Only perform an AVS check, do not store as a transaction. Possible values: "Y", "N"
		 */
		public function SetAvsOnly($avsOnly)
		{
				$this->avsOnly = $avsOnly;
		}
	   /**

		 *  Setter forPO_SCHEMA_FILE MPI Reference code
		 *
		 *  @param mpiRef MPI Reference code supplied by Nuvei MPI redirect
		 */
		public function SetMpiRef($mpiRef)
		{
				$this->mpiRef = $mpiRef;
		}

        public function SetXid($xid)
        {
            $this->xid = $xid;
        }

        public function SetCavv($cavv)
        {
            $this->cavv = $cavv;
        }

	   /**
		 *  Setter for Mobile Number
		 *
		 *  @param mobileNumber Mobile Number of cardholder. If sent an SMS receipt will be sent to them
		 */
		public function SetMobileNumber($mobileNumber)
		{
				$this->mobileNumber = $mobileNumber;
		}
	   /**
		 *  Setter for Device ID
		 *
		 *  @param deviceId Device ID to identify this source to the XML gateway
		 */
		public function SetDeviceId($deviceId)
		{
				$this->deviceId = $deviceId;
		}
	   /**
		 *  Setter for Phone number
		 *
		 *  @param phone Phone number of cardholder
		 */
		public function SetPhone($phone)
		{
				$this->phone = $phone;
		}
	   /**
		 *  Setter for the cardholders IP address
		 *
		 *  @param ipAddress IP Address of the cardholder
		 */
		public function SetIPAddress($ipAddress)
		{
            $this->ipAddress = $ipAddress;
        }
        /**
         *  Setter for City
         *
         *  @param city Cardholders City
         */
        public function SetCity($city)
        {
            $this->city = $city;
        }
        /**
         *  Setter for Region
         *
         *  @param region Cardholders Region
         */
        public function SetRegion($region)
        {
            $this->region = $region;
        }
        /**
         *  Setter for Country
         *
         *  @param country Cardholders Country
         */
        public function SetCountry($country)
        {
            $this->country = $country;
        }
	   /**
		 *  Setter for multi-currency value
		 *  This is required to be set for multi-currency terminals because the Hash is calculated differently.
		 */
		public function SetMultiCur()
		{
				$this->multicur = true;
		}
	   /**
		 *  Setter to flag transaction as a Mail Order. If not set the transaction defaults to eCommerce
		 */
		public function SetMotoTrans()
		{
				$this->terminalType = "1";
				$this->transactionType = "4";
		}
	   /**
		 *  Setter to flag transaction as a Mail Order. If not set the transaction defaults to eCommerce
		 */
		public function SetTrackData($trackData)
		{
				$this->terminalType = "3";
				$this->transactionType = "0";
				$this->cardNumber = "";
				$this->trackData = $trackData;
		}
        /**
         *  Setter for transactionType
         *
         *  @param transactionType
         */
        public function SetTransactionType($transactionType)
        {
            $this->transactionType = $transactionType;
        }
	   /**
		 *  Setter for hash value
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by Nuvei or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 */
		public function SetHash($sharedSecret)
		{
				if(isset($this->multicur) && $this->multicur == true) $this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->currency . $this->amount . $this->dateTime . $sharedSecret);
				else $this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->amount . $this->dateTime . $sharedSecret);
		}

		public function ProcessRequestToGateway($sharedSecret, $serverUrl)
		{
				$this->SetHash($sharedSecret);
				$headers = array();
				$headers['terminal'] = $this->terminalId;
				$responseString = $this->SendRequestToGateway($this->GenerateXml(), $serverUrl, $headers);
				$response = new NuveiGatewayXmlAuthResponse($responseString);
				return $response;
		}

		public function GenerateXml()
		{
				$requestXML = new DOMDocument("1.0");
				$requestXML->formatOutput = true;

				$requestString = $requestXML->createElement("PAYMENT");
				$requestXML->appendChild($requestString);

				$node = $requestXML->createElement("ORDERID");
				$node->appendChild($requestXML->createTextNode($this->orderId));
				$requestString->appendChild($node);

				$node = $requestXML->createElement("TERMINALID");
				$requestString->appendChild($node);
				$node->appendChild($requestXML->createTextNode($this->terminalId));

				$node = $requestXML->createElement("AMOUNT");
				$requestString->appendChild($node);
				$node->appendChild($requestXML->createTextNode($this->amount));

				$node = $requestXML->createElement("DATETIME");
				$requestString->appendChild($node);
				$node->appendChild($requestXML->createTextNode($this->dateTime));

				if($this->trackData !== NULL)
				{
					$node = $requestXML->createElement("TRACKDATA");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->trackData));
				} else {
					$node = $requestXML->createElement("CARDNUMBER");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->cardNumber));
				}

				$node = $requestXML->createElement("CARDTYPE");
				$requestString->appendChild($node);
				$node->appendChild($requestXML->createTextNode($this->cardType));

				if($this->cardExpiry !== NULL && $this->cardHolderName !== NULL && $this->trackData == NULL)
				{
					$node = $requestXML->createElement("CARDEXPIRY");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->cardExpiry));

					$node = $requestXML->createElement("CARDHOLDERNAME");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->cardHolderName));
				}

				$node = $requestXML->createElement("HASH");
				$requestString->appendChild($node);
				$node->appendChild($requestXML->createTextNode($this->hash));

				$node = $requestXML->createElement("CURRENCY");
				$requestString->appendChild($node);
				$node->appendChild($requestXML->createTextNode($this->currency));

				if($this->foreignCurInfoSet)
				{
					$dcNode = $requestXML->createElement("FOREIGNCURRENCYINFORMATION");
					$requestString->appendChild($dcNode );

					$dcSubNode = $requestXML->createElement("CARDCURRENCY");
					$dcSubNode ->appendChild($requestXML->createTextNode($this->cardCurrency));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXML->createElement("CARDAMOUNT");
					$dcSubNode ->appendChild($requestXML->createTextNode($this->cardAmount));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXML->createElement("CONVERSIONRATE");
					$dcSubNode ->appendChild($requestXML->createTextNode($this->conversionRate));
					$dcNode->appendChild($dcSubNode);
				}

				$node = $requestXML->createElement("TERMINALTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXML->createTextNode($this->terminalType);
				$node->appendChild($nodeText);

				$node = $requestXML->createElement("TRANSACTIONTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXML->createTextNode($this->transactionType);
				$node->appendChild($nodeText);

				if($this->autoReady !== NULL)
				{
					$node = $requestXML->createElement("AUTOREADY");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->autoReady));
				}

				if($this->email !== NULL)
				{
					$node = $requestXML->createElement("EMAIL");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->email));
				}

				if($this->cvv !== NULL)
				{
					$node = $requestXML->createElement("CVV");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->cvv));
				}

				if($this->issueNo !== NULL)
				{
					$node = $requestXML->createElement("ISSUENO");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->issueNo));
				}

				if($this->postCode !== NULL)
				{
					if($this->address1 !== NULL)
					{
						$node = $requestXML->createElement("ADDRESS1");
						$requestString->appendChild($node);
						$node->appendChild($requestXML->createTextNode($this->address1));
					}
					if($this->address2 !== NULL)
					{
						$node = $requestXML->createElement("ADDRESS2");
						$requestString->appendChild($node);
						$node->appendChild($requestXML->createTextNode($this->address2));
					}

					$node = $requestXML->createElement("POSTCODE");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->postCode));
				}

				if($this->avsOnly !== NULL)
				{
					$node = $requestXML->createElement("AVSONLY");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->avsOnly));
				}

				if($this->description !== NULL)
				{
					$node = $requestXML->createElement("DESCRIPTION");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->description));
				}

                if($this->mpiRef !== NULL)
                {
                    $node = $requestXML->createElement("MPIREF");
                    $requestString->appendChild($node);
                    $node->appendChild($requestXML->createTextNode($this->mpiRef));
                }

                if($this->xid !== NULL)
                {
                    $node = $requestXML->createElement("XID");
                    $requestString->appendChild($node);
                    $node->appendChild($requestXML->createTextNode($this->xid));
                }

                if($this->cavv !== NULL)
                {
                    $node = $requestXML->createElement("CAVV");
                    $requestString->appendChild($node);
                    $node->appendChild($requestXML->createTextNode($this->cavv));
                }

				if($this->mobileNumber !== NULL)
				{
					$node = $requestXML->createElement("MOBILENUMBER");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->mobileNumber));
				}

				if($this->deviceId !== NULL)
				{
					$node = $requestXML->createElement("DEVICEID");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->deviceId));
				}

				if($this->phone !== NULL)
				{
					$node = $requestXML->createElement("PHONE");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->phone));
				}

                if($this->city !== NULL)
                {
                    $node = $requestXML->createElement("CITY");
                    $requestString->appendChild($node);
                    $node->appendChild($requestXML->createTextNode($this->city));
                }

                if($this->region !== NULL)
                {
                    $node = $requestXML->createElement("REGION");
                    $requestString->appendChild($node);
                    $node->appendChild($requestXML->createTextNode($this->region));
                }

                if($this->country !== NULL)
                {
                    $node = $requestXML->createElement("COUNTRY");
                    $requestString->appendChild($node);
                    $node->appendChild($requestXML->createTextNode($this->country));
                }

				if($this->ipAddress !== NULL)
				{
					$node = $requestXML->createElement("IPADDRESS");
					$requestString->appendChild($node);
					$node->appendChild($requestXML->createTextNode($this->ipAddress));
				}

                return $requestXML->saveXML();
		}
}

/**
  *  Holder class for parsed response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  *
  *  IsError should be checked before accessing any fields.
  *
  *  ErrorString will contain the error if one occurred.
  */
class NuveiGatewayXmlAuthResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}

		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $responseCode;
		public function ResponseCode()
		{
				return $this->responseCode;
		}

		private $responseText;
		public function ResponseText()
		{
				return $this->responseText;
		}

		private $approvalCode;
		public function ApprovalCode()
		{
				return $this->approvalCode;
		}

		private $authorizedAmount;
		public function AuthorizedAmount()
		{
				return $this->authorizedAmount;
		}

		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}

		private $avsResponse;
		public function AvsResponse()
		{
				return $this->avsResponse;
		}

		private $cvvResponse;
		public function CvvResponse()
		{
				return $this->cvvResponse;
		}

		private $uniqueRef;
		public function UniqueRef()
		{
				return $this->uniqueRef;
		}

		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function __construct($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "PAYMENTRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("PAYMENTRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->uniqueRef = $node->getElementsByTagName('UNIQUEREF')->item(0)->nodeValue;
									$this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
									$this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
									$this->approvalCode = $node->getElementsByTagName('APPROVALCODE')->item(0)->nodeValue;
									$this->authorizedAmount = $node->getElementsByTagName('AUTHORIZEDAMOUNT')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->avsResponse = $node->getElementsByTagName('AVSRESPONSE')->item(0)->nodeValue;
									$this->cvvResponse = $node->getElementsByTagName('CVVRESPONSE')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}



/**
 *  Used for processing XML Refund Authorisations through the Nuvei XML Gateway.
 *
 *  Basic request is configured on initialisation. There are no coptional fields.
 */
class NuveiGatewayXmlRefundRequest extends NuveiGatewayRequest
{
    private $terminalId;
    private $orderId;
    private $uniqueRef;
    private $amount;
    public function Amount()
    {
        return $this->amount;
    }
    private $dateTime;
    private $hash;
    private $operator;
    private $reason;
    private $autoReady;

    /**
     *  Creates the refund request for processing an XML Transaction
     *  through the Nuvei XML Gateway
     *
     *  @param terminalId Terminal ID provided by Nuvei
     *  @param orderId A unique merchant identifier. Alpha numeric and max size 12 chars.
     *  @param currency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
     *  @param amount Transaction Amount, Double formatted to 2 decimal places.
     *  @param operator An identifier for who executed this transaction
     *  @param reason The reason for the refund
     */
    public function __construct($terminalId,
                                $orderId,
                                $amount,
                                $operator,
                                $reason)
    {
        $this->dateTime = $this->GetFormattedDate();
        $this->amount = $amount;
        $this->terminalId = $terminalId;
        $this->orderId = $orderId;
        $this->operator = $operator;
        $this->reason = $reason;
    }
    /**
     *  Setter for UniqueRef

     *
     *  @param uniqueRef
     *  Unique Reference of transaction returned from gateway in authorisation response
     */
    public function SetUniqueRef($uniqueRef)
    {
        $this->uniqueRef = $uniqueRef;
        $this->orderId = "";
    }
    /**
     *  Setter for Auto Ready Value

     *
     *  @param autoReady
     *  Auto Ready is an optional parameter and defines if the transaction should be settled automatically.
     *
     *  Accepted Values :

     *
     *  Y   -   Transaction will be settled in next batch
     *  N   -   Transaction will not be settled until user changes state in Merchant Selfcare Section
     */
    public function SetAutoReady($autoReady)
    {
        $this->autoReady = $autoReady;
    }
    /**
     *  Setter for hash value
     *
     *  @param sharedSecret
     *  Shared secret either supplied by Nuvei or configured under
     *  Terminal Settings in the Merchant Selfcare System.
     */
    public function SetHash($sharedSecret)
    {
        if($this->uniqueRef !== NULL)
        {
            $this->hash = $this->GetRequestHash($this->terminalId . $this->uniqueRef . $this->amount . $this->dateTime . $sharedSecret);
        } else {
            $this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->amount . $this->dateTime . $sharedSecret);
        }
    }

    public function ProcessRequestToGateway($sharedSecret, $serverUrl)
    {
        $this->SetHash($sharedSecret);
        $headers = array();
        $headers['terminal'] = $this->terminalId;
        $responseString = $this->SendRequestToGateway($this->GenerateXml(), $serverUrl, $headers);
        $response = new NuveiGatewayXmlRefundResponse($responseString);
        return $response;
    }
    public function GenerateXml()
    {
        $requestXML = new DOMDocument("1.0");
        $requestXML->formatOutput = true;

        $requestString = $requestXML->createElement("REFUND");
        $requestXML->appendChild($requestString);

        if($this->uniqueRef !== NULL)
        {
            $node = $requestXML->createElement("UNIQUEREF");
            $node->appendChild($requestXML->createTextNode($this->uniqueRef));
            $requestString->appendChild($node);
        } else {
            $node = $requestXML->createElement("ORDERID");
            $node->appendChild($requestXML->createTextNode($this->orderId));
            $requestString->appendChild($node);
        }

        $node = $requestXML->createElement("TERMINALID");
        $node->appendChild($requestXML->createTextNode($this->terminalId));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("AMOUNT");
        $node->appendChild($requestXML->createTextNode($this->amount));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("DATETIME");
        $node->appendChild($requestXML->createTextNode($this->dateTime));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("HASH");
        $node->appendChild($requestXML->createTextNode($this->hash));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("OPERATOR");
        $node->appendChild($requestXML->createTextNode($this->operator));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("REASON");
        $node->appendChild($requestXML->createTextNode($this->reason));
        $requestString->appendChild($node);

        if($this->autoReady !== NULL)
        {
            $node = $requestXML->createElement("AUTOREADY");
            $requestString->appendChild($node);
            $node->appendChild($requestXML->createTextNode($this->autoReady));
        }

        return $requestXML->saveXML();

    }
}

/**
 *  Holder class for parsed response. If there was an error there will be an error string
 *  otherwise all values will be populated with the parsed payment response values.
 *
 *  IsError should be checked before accessing any fields.
 *
 *  ErrorString will contain the error if one occurred.
 */
class NuveiGatewayXmlRefundResponse
{
    private $isError = false;
    public function IsError()
    {
        return $this->isError;
    }

    private $errorString;
    public function ErrorString()
    {
        return $this->errorString;
    }

    private $responseCode;
    public function ResponseCode()
    {
        return $this->responseCode;
    }

    private $responseText;
    public function ResponseText()
    {
        return $this->responseText;
    }

    private $approvalCode;
    public function OrderId()
    {
        return $this->orderId;
    }

    private $dateTime;
    public function DateTime()
    {
        return $this->dateTime;
    }

    private $uniqueRef;
    public function UniqueRef()
    {
        return $this->uniqueRef;
    }

    private $hash;
    public function Hash()
    {
        return $this->hash;
    }

    public function __construct($responseXml)
    {
        $doc = new DOMDocument();
        $doc->loadXML($responseXml);
        try
        {
            if (strpos($responseXml, "ERROR"))
            {
                $responseNodes = $doc->getElementsByTagName("ERROR");
                foreach( $responseNodes as $node )
                {
                    $this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
                }
                $this->isError = true;
            }
            else if (strpos($responseXml, "REFUNDRESPONSE"))
            {
                $responseNodes = $doc->getElementsByTagName("REFUNDRESPONSE");

                foreach( $responseNodes as $node )
                {
                    $this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
                    $this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
                    $this->uniqueRef = $node->getElementsByTagName('UNIQUEREF')->item(0)->nodeValue;
                    $this->orderId = $node->getElementsByTagName('ORDERID')->item(0)->nodeValue;
                    $this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
                    $this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
                }
            }
            else
            {
                throw new Exception("Invalid Response");
            }
        }
        catch (Exception $e)
        {
            $this->isError = true;
            $this->errorString = $e->getMessage();
        }
    }
}


class NuveiGatewayXmlTerminalFeaturesRequest extends NuveiGatewayRequest
{
    private $terminalId;
    private $customFieldLanguage;
    private $dateTime;
    private $hash;

    public function __construct($terminalId, $customFieldLanguage)
    {
        $this->terminalId = $terminalId;
        $this->customFieldLanguage = $customFieldLanguage;
        $this->dateTime = $this->GetFormattedDate();
    }

    public function SetHash($sharedSecret)
    {
        $this->hash = $this->GetRequestHash($this->terminalId . $this->customFieldLanguage . $this->dateTime . $sharedSecret);
    }

    public function ProcessRequestToGateway($sharedSecret, $serverUrl)
    {
        $this->SetHash($sharedSecret);
        $headers = array();
        $headers['terminal'] = $this->terminalId;
        $responseString = $this->SendRequestToGateway($this->GenerateXml(), $serverUrl, $headers);
        $response = new NuveiGatewayXmlTerminalFeaturesResponse($responseString);
        return $response;
    }

    public function GenerateXml()
    {
        $requestXML = new DOMDocument("1.0");
        $requestXML->formatOutput = true;

        $requestString = $requestXML->createElement("TERMINAL_CONFIGURATION");
        $requestXML->appendChild($requestString);

        $node = $requestXML->createElement("TERMINALID");
        $node->appendChild($requestXML->createTextNode($this->terminalId));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("CUSTOM_FIELD_LANGUAGE");
        $node->appendChild($requestXML->createTextNode($this->customFieldLanguage));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("DATETIME");
        $node->appendChild($requestXML->createTextNode($this->dateTime));
        $requestString->appendChild($node);

        $node = $requestXML->createElement("HASH");
        $node->appendChild($requestXML->createTextNode($this->hash));
        $requestString->appendChild($node);

        return $requestXML->saveXML();
    }
}
class NuveiGatewayXmlTerminalFeaturesResponse
{
    private $isError = false;
    public function IsError()
    {
        return $this->isError;
    }

    private $errorString;
    public function ErrorString()
    {
        return $this->errorString;
    }

    private $hash;
    public function Hash()
    {
        return $this->hash;
    }

    private $settings;
    public function getSettings()
    {
        return $this->settings;
    }

    public function getNodeTree($node) {
        $settings = array();

        foreach ($node->childNodes as $child) {
            if (!$this->hasChild($child)) {
                if (!isset($settings[$child->nodeName]))
                    $settings[$child->nodeName] = $child->nodeValue;
                else {
                    if(!is_array($settings[$child->nodeName])) {
                        $settings[$child->nodeName] = [$settings[$child->nodeName]];
                        array_push($settings[$child->nodeName], $child->nodeValue);
                    }
                    else
                        array_push($settings[$child->nodeName], $child->nodeValue);
                }
            }
            else {
                if (!isset($settings[$child->nodeName])) {
                    $settings[$child->nodeName] = array();

                    $settings[$child->nodeName] = $this->getNodeTree($child);
                } else {
                    if(!isset($settings[$child->nodeName][0]))
                        $settings[$child->nodeName] = [$settings[$child->nodeName]];

                    array_push($settings[$child->nodeName], $this->getNodeTree($child));
                }
            }
        }

        return $settings;
    }


    public function hasChild($p)
    {
        if ($p->hasChildNodes()) {
            foreach ($p->childNodes as $c) {
                if ($c->nodeType == XML_ELEMENT_NODE)
                    return true;
            }
        }
    }

    public function __construct($responseXml)
    {
        $doc = new DOMDocument();
        $doc->loadXML($responseXml);
        try
        {
            if (strpos($responseXml, "ERROR"))
            {
                $responseNodes = $doc->getElementsByTagName("ERROR");
                foreach( $responseNodes as $node )
                {
                    $this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
                }
                $this->isError = true;
            }
            else if (strpos($responseXml, "TERMINAL_CONFIGURATION_RESPONSE"))
            {
                $responseNodes = $doc->getElementsByTagName("TERMINAL_CONFIGURATION_RESPONSE");

                foreach( $responseNodes as $node )
                {
                    foreach($node->childNodes as $child) {
                        if(!$this->hasChild($child))
                            $this->settings[$child->nodeName] = $child->nodeValue;
                        else {
                            if(!isset($this->settings[$child->nodeName]))
                                $this->settings[$child->nodeName] = array();

                            $this->settings[$child->nodeName] = $this->getNodeTree($child);
                        }
                    }
                }
            }
            else
            {
                throw new Exception("Invalid Response");
            }
        }
        catch (Exception $e)
        {
            $this->isError = true;
            $this->errorString = $e->getMessage();
        }
    }
}

?>