<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once( dirname( __FILE__ ) . '/nuvei_gateway_xml_api.php' );

/**
 * Nuvei_Gateway class.
 *
 * @extends WC_Payment_Gateway
 */
class Nuvei_Gateway extends WC_Payment_Gateway_CC {

	/**
	 * Should we capture Credit cards
	 *
	 * @var bool
	 */
	public $capture;

	/**
	 * Alternate credit card statement name
	 *
	 * @var bool
	 */
	public $statement_descriptor;

	/**
	 * Checkout enabled
	 *
	 * @var bool
	 */
	public $nuvei_checkout;

	/**
	 * Checkout Locale
	 *
	 * @var string
	 */
	public $nuvei_checkout_locale;

	/**
	 * Credit card image
	 *
	 * @var string
	 */
	public $nuvei_checkout_image;

	/**
	 * Should we store the users credit cards?
	 *
	 * @var bool
	 */
	public $saved_cards;

	/**
	 * API access secret key
	 *
	 * @var string
	 */
	public $secret_key;

	/**
	 * Api access publishable key
	 *
	 * @var string
	 */
	public $publishable_key;

	/**
	 * Is test mode active?
	 *
	 * @var bool
	 */
	public $testmode;

	/**
	 * Logging enabled?
	 *
	 * @var bool
	 */
	public $logging;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->id                   = 'nuvei';
		$this->method_title         = __( 'Nuvei', 'woocommerce-gateway-nuvei' );
		$this->method_description   = sprintf( __( 'Nuvei works by adding credit card fields on the checkout and then sending the details to gateway for verification.', 'woocommerce-gateway-nuvei' ));
		$this->has_fields           = true;
		$this->supports             = array(
			'products',
			'refunds',
            //'subscriptions',
			//'subscription_cancellation',
			//'subscription_reactivation',
			//'subscription_suspension',
			//'subscription_amount_changes',
			//'subscription_payment_method_change', // Subs 1.n compatibility.
			//'subscription_payment_method_change_customer',
			//'subscription_payment_method_change_admin',
			//'subscription_date_changes',
			//'multiple_subscriptions',
			'pre-orders',
			//'tokenization',
			'add_payment_method',
		);

		//TODO: refactor to use correct hooks in a later release
        global $post;
        $order_id = $post->ID;
        if($order_id) {
            $order = wc_get_order( $order_id );
            if ($order) {
                $orderStatus = $order->get_status();

                if (!$order->get_transaction_id() || $orderStatus === 'refunded') {
                    unset($this->supports[1]);
                }
            }
        }

		// Load the form fields.
		$this->init_form_fields();

		// Load the settings.
		$this->init_settings();

		// Get setting values.
		$this->title                   = $this->get_option( 'title' );
		$this->description             = $this->get_option( 'description' );
		$this->enabled                 = $this->get_option( 'enabled' );
        $this->testmode                = 'yes' === $this->get_option( 'testmode' );
        $this->avs                = 'yes' === $this->get_option( 'avs' );
        $this->enabled3ds                = 'yes' === $this->get_option( '3ds' );
		$this->capture                 = 'yes' === $this->get_option( 'capture', 'yes' );
		$this->statement_descriptor    = $this->get_option( 'statement_descriptor', wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES ) );
		$this->nuvei_checkout         = 'yes' === $this->get_option( 'nuvei_checkout' );
		$this->nuvei_integration_type = $this->get_option( 'nuvei_integration_type' );
		$this->nuvei_checkout_locale  = $this->get_option( 'nuvei_checkout_locale' );
		$this->nuvei_checkout_image   = $this->get_option( 'nuvei_checkout_image', '' );
		$this->saved_cards             = 'yes' === $this->get_option( 'saved_cards' );
		$this->secret_key              = $this->testmode ? $this->get_option( 'test_secret_key' ) : $this->get_option( 'live_secret_key' );
		$this->publishable_key         = $this->testmode ? $this->get_option( 'test_publishable_key' ) : $this->get_option( 'live_publishable_key' );
		$this->currency         = $this->testmode ? $this->get_option( 'test_currency' ) : $this->get_option( 'live_currency' );
		$this->multicurrency         = $this->testmode ? $this->get_option( 'test_multicurrency' ) : $this->get_option( 'live_multicurrency' );
		$this->secret_key2              = $this->testmode ? $this->get_option( 'test_secret_key2' ) : $this->get_option( 'live_secret_key2' );
		$this->publishable_key2         = $this->testmode ? $this->get_option( 'test_publishable_key2' ) : $this->get_option( 'live_publishable_key2' );
		$this->currency2         = $this->testmode ? $this->get_option( 'test_currency2' ) : $this->get_option( 'live_currency2' );
		$this->multicurrency2         = $this->testmode ? $this->get_option( 'test_multicurrency2' ) : $this->get_option( 'live_multicurrency2' );
		$this->secret_key3              = $this->testmode ? $this->get_option( 'test_secret_key3' ) : $this->get_option( 'live_secret_key3' );
		$this->publishable_key3         = $this->testmode ? $this->get_option( 'test_publishable_key3' ) : $this->get_option( 'live_publishable_key3' );
		$this->currency3         = $this->testmode ? $this->get_option( 'test_currency3' ) : $this->get_option( 'live_currency3' );
		$this->multicurrency3         = $this->testmode ? $this->get_option( 'test_multicurrency3' ) : $this->get_option( 'live_multicurrency3' );
		$this->logging                 = 'yes' === $this->get_option( 'logging' );
        $this->terminal_settings = [];

		if ( $this->nuvei_integration_type=='hpp') {
			$this->order_button_text = __( 'Continue to payment', 'woocommerce-gateway-nuvei' );
		}

		if ( $this->testmode ) {
			$this->description .= ' ' . sprintf( __( ' - TEST MODE ENABLED') );
			$this->description  = trim( $this->description );
		}

		// Hooks.
		add_action( 'wp_enqueue_scripts', array( $this, 'payment_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_before_checkout_form', array( $this, 'display_errors' ) );
	}

    public function display_errors() {
        if(isset($_GET['ERRORSTRING']))
            wc_add_notice( sanitize_text_field($_GET['ERRORSTRING']), 'error' );
    }

    /**
     * Outputs fields for entering credit card information.
     * @since 2.6.0
     */
    public function form() {
        session_start();
        if (version_compare(PHP_VERSION, '5.3.0') >= 0) {
            if (empty($_SESSION['CSRF-Token'])) {
                $_SESSION['CSRF-Token'] = bin2hex(random_bytes(32));
            }
        } elseif (version_compare(PHP_VERSION, '5.3.0') >= 0) {
            if (empty($_SESSION['CSRF-Token'])) {
                if (function_exists('mcrypt_create_iv')) {
                    $_SESSION['CSRF-Token'] = bin2hex(mcrypt_create_iv(32, MCRYPT_DEV_URANDOM));
                } else {
                    $_SESSION['CSRF-Token'] = bin2hex(openssl_random_pseudo_bytes(32));
                }
            }
        }

        $cardholder = '';

        if(isset($_GET['key'])) {
            $order_id = wc_get_order_id_by_order_key( urldecode( $_GET['key'] ) );
            $order    = wc_get_order( $order_id );

            $firstName = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_first_name : $order->get_billing_first_name();
            $lastName = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_last_name : $order->get_billing_last_name();
            $cardholder = $firstName . ' ' . $lastName;
        }


        if(WC()->session->get( 'request3ds')) {
            $request3ds = WC()->session->get( 'request3ds');

            echo '<script type="text/javascript">
                    var form = document.createElement("form");
                    form.style.display = "none";
                    
                    var TERMINALID = document.createElement("input"); 
                    TERMINALID.value="'. $request3ds['TERMINALID'] .'";
                    TERMINALID.name="TERMINALID";
                    form.appendChild(TERMINALID);  
                    
                    var CARDHOLDERNAME = document.createElement("input"); 
                    CARDHOLDERNAME.value="'. $request3ds['CARDHOLDERNAME'] .'";
                    CARDHOLDERNAME.name="CARDHOLDERNAME";
                    form.appendChild(CARDHOLDERNAME);  
                    
                    var CARDNUMBER = document.createElement("input"); 
                    CARDNUMBER.value="'. $request3ds['CARDNUMBER'] .'";
                    CARDNUMBER.name="CARDNUMBER";
                    form.appendChild(CARDNUMBER);  
                    
                    var CARDEXPIRY = document.createElement("input"); 
                    CARDEXPIRY.value="'. $request3ds['CARDEXPIRY'] .'";
                    CARDEXPIRY.name="CARDEXPIRY";
                    form.appendChild(CARDEXPIRY);  
                    
                    var CARDTYPE = document.createElement("input"); 
                    CARDTYPE.value="'. $request3ds['CARDTYPE'] .'";
                    CARDTYPE.name="CARDTYPE";
                    form.appendChild(CARDTYPE);  
                    
                    var AMOUNT = document.createElement("input"); 
                    AMOUNT.value="'. $request3ds['AMOUNT'] .'";
                    AMOUNT.name="AMOUNT";
                    form.appendChild(AMOUNT);  
                    
                    var CURRENCY = document.createElement("input"); 
                    CURRENCY.value="'. $request3ds['CURRENCY'] .'";
                    CURRENCY.name="CURRENCY";
                    form.appendChild(CURRENCY);  
                    
                    var ORDERID = document.createElement("input"); 
                    ORDERID.value="'. $request3ds['ORDERID'] .'";
                    ORDERID.name="ORDERID";
                    form.appendChild(ORDERID);  
                    
                    var CVV = document.createElement("input"); 
                    CVV.value="'. $request3ds['CVV'] .'";
                    CVV.name="CVV";
                    form.appendChild(CVV);  
                    
                    var DATETIME = document.createElement("input"); 
                    DATETIME.value="'. $request3ds['DATETIME'] .'";
                    DATETIME.name="DATETIME";
                    form.appendChild(DATETIME);  
                    
                    var HASH = document.createElement("input"); 
                    HASH.value="'. $request3ds['HASH'] .'";
                    HASH.name="HASH";
                    form.appendChild(HASH);  
                                    
                    form.method = "POST";
                    form.action = "'. $request3ds['redirectUrl'] .'";  
                
                    document.body.appendChild(form);
                
                    form.submit();
                  </script>';

            WC()->session->__unset( 'request3ds');

            return;
        }

        wp_enqueue_script( 'wc-credit-card-form' );

        $fields = array();

        $cvc_field = '<p class="form-row form-row-last">
			<label for="' . esc_attr( $this->id ) . '-card-cvc">' . esc_html__( 'Card code', 'woocommerce' ) . ' <span class="required">*</span></label>
			<input id="' . esc_attr( $this->id ) . '-card-cvc" class="input-text wc-credit-card-form-card-cvc" inputmode="numeric" autocomplete="off" autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" maxlength="4" placeholder="' . esc_attr__( 'CVC', 'woocommerce' ) . '" ' . $this->field_name( 'card-cvc' ) . ' style="width:100px" />
		</p>';

        $default_fields = array(
            'card-name-field' => '<p class="form-row form-row-wide">
				<label for="' . esc_attr( $this->id ) . '-card-name">' . esc_html__( 'Card holder', 'woocommerce' ) . ' <span class="required">*</span></label>
				<input id="' . esc_attr( $this->id ) . '-card-name" class="input-text wc-credit-card-form-card-name" inputmode="text" autocomplete="cc-name" autocorrect="no" autocapitalize="no" spellcheck="no" type="text" placeholder="" ' . $this->field_name( 'card-name' ) . ' value="'.$cardholder.'" />
			</p>',
            'card-number-field' => '<p class="form-row form-row-wide">
				<label for="' . esc_attr( $this->id ) . '-card-number">' . esc_html__( 'Card number', 'woocommerce' ) . ' <span class="required">*</span></label>
				<input id="' . esc_attr( $this->id ) . '-card-number" class="input-text wc-credit-card-form-card-number" inputmode="numeric" autocomplete="cc-number" autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" placeholder="&bull;&bull;&bull;&bull; &bull;&bull;&bull;&bull; &bull;&bull;&bull;&bull; &bull;&bull;&bull;&bull;" ' . $this->field_name( 'card-number' ) . ' />
			</p>',
            'card-expiry-field' => '<p class="form-row form-row-first">
				<label for="' . esc_attr( $this->id ) . '-card-expiry">' . esc_html__( 'Expiry (MM/YY)', 'woocommerce' ) . ' <span class="required">*</span></label>
				<input id="' . esc_attr( $this->id ) . '-card-expiry" class="input-text wc-credit-card-form-card-expiry" inputmode="numeric" autocomplete="cc-exp" autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" placeholder="' . esc_attr__( 'MM / YY', 'woocommerce' ) . '" ' . $this->field_name( 'card-expiry' ) . ' />
			</p>',
            'csrf-token' => '<p class="form-row form-row-first" style="display: none;">
				<input id="' . esc_attr( $this->id ) . '-csrf-token" class="input-text wc-credit-card-form-card-expiry" inputmode="hidden" autocorrect="no" autocapitalize="no" spellcheck="no" value="' . $_SESSION['CSRF-Token'] . '" />
			</p>',
        );

        if ( ! $this->supports( 'credit_card_form_cvc_on_saved_method' ) ) {
            $default_fields['card-cvc-field'] = $cvc_field;
        }

        $fields = wp_parse_args( $fields, apply_filters( 'woocommerce_credit_card_form_fields', $default_fields, $this->id ) );
        ?>

        <fieldset id="wc-<?php echo esc_attr( $this->id ); ?>-cc-form" class='wc-credit-card-form wc-payment-form'>
            <?php do_action( 'woocommerce_credit_card_form_start', $this->id ); ?>
            <?php
            foreach ( $fields as $field ) {
                echo $field;
            }
            ?>
            <?php do_action( 'woocommerce_credit_card_form_end', $this->id ); ?>
            <div class="clear"></div>
        </fieldset>
        <?php

        if ( $this->supports( 'credit_card_form_cvc_on_saved_method' ) ) {
            echo '<fieldset>' . $cvc_field . '</fieldset>';
        }
    }

    /**
     * Get_icon function.
     *
     * @access public
     * @return string
     */
    public function get_icon() {
        $this->get_terminal_settings();

        $style = version_compare( WC()->version, '2.6', '>=' ) ? 'style="margin-left: 0.3em"' : '';

        $icon = '';

        if(isset($this->terminal_settings['SUPPORTED_CARDTYPES'])) {
            if(is_array($this->terminal_settings['SUPPORTED_CARDTYPES']['CARDTYPE']))
                $cards = $this->terminal_settings['SUPPORTED_CARDTYPES']['CARDTYPE'];
            else
                $cards = [ $this->terminal_settings['SUPPORTED_CARDTYPES']['CARDTYPE'] ];

            if(in_array('Visa Credit', $cards) || in_array('Visa Debit', $cards) || in_array('Visa Electron', $cards) || in_array('Delta', $cards))
                $icon .= '<img src="' . WC_Nuvei_PLUGIN_URL . '/assets/images/icons/credit-cards/visa.svg" alt="Visa" width="32" ' . $style . ' />';

            if(in_array('MasterCard', $cards) || in_array('Debit MasterCard', $cards) || in_array('Maestro', $cards))
                $icon .= '<img src="' . WC_Nuvei_PLUGIN_URL . '/assets/images/icons/credit-cards/mastercard.svg" alt="Mastercard" width="32" ' . $style . ' />';

            if(in_array('American Express', $cards))
                $icon .= '<img src="' . WC_Nuvei_PLUGIN_URL . '/assets/images/icons/credit-cards/amex.svg" alt="Amex" width="32" ' . $style . ' />';

            if(in_array('Diners', $cards) || in_array('Discover', $cards))
                $icon .= '<img src="' . WC_Nuvei_PLUGIN_URL . '/assets/images/icons/credit-cards/diners.svg" alt="Diners" width="32" ' . $style . ' />';

            if(in_array('JCB', $cards))
                $icon .= '<img src="' . WC_Nuvei_PLUGIN_URL . '/assets/images/icons/credit-cards/jcb.svg" alt="JCB" width="32" ' . $style . ' />';

            if($icon === '')
                $icon .= '<img src="' . WC_Nuvei_PLUGIN_URL . '/assets/images/icons/credit-cards/generic.svg" alt="Debit or Credit Card" width="32" ' . $style . ' />';

        }


        return apply_filters( 'woocommerce_gateway_icon', $icon, $this->id );
    }

	/**
	 * Get Nuvei amount to pay
	 *
	 * @param float  $total Amount due.
	 * @param string $currency Accepted currency.
	 *
	 * @return float|int
	 */
	public function get_nuvei_amount( $total, $currency = '' ) {
		if ( ! $currency ) {
			$currency = get_woocommerce_currency();
		}
		switch ( strtoupper( $currency ) ) {
			// Zero decimal currencies.
			case 'BIF' :
			case 'CLP' :
			case 'DJF' :
			case 'GNF' :
			case 'JPY' :
			case 'KMF' :
			case 'KRW' :
			case 'MGA' :
			case 'PYG' :
			case 'RWF' :
			case 'VND' :
			case 'VUV' :
			case 'XAF' :
			case 'XOF' :
			case 'XPF' :
				$total = absint( $total );
				break;
			default :
				$total = round( $total, 2 ) * 100; // In cents.
				break;
		}
		return $total;
	}

	/**
	 * Check if SSL is enabled and notify the user
	 */
	public function admin_notices() {
		if ( 'no' === $this->enabled ) {
			return;
		}

		// Show message if enabled and FORCE SSL is disabled and WordpressHTTPS plugin is not detected.
		if ( ( function_exists( 'wc_site_is_https' ) && ! wc_site_is_https() ) && ( 'no' === get_option( 'woocommerce_force_ssl_checkout' ) && ! class_exists( 'WordPressHTTPS' ) ) ) {
			echo '<div class="error nuvei-ssl-message"><p>' . sprintf( __( 'Nuvei is enabled, but the <a href="%1$s">force SSL option</a> is disabled; your checkout may not be secure! Please enable SSL and ensure your server has a valid <a href="%2$s" target="_blank">SSL certificate</a> - Nuvei will only work in test mode.', 'woocommerce-gateway-nuvei' ), admin_url( 'admin.php?page=wc-settings&tab=checkout' ), 'https://en.wikipedia.org/wiki/Transport_Layer_Security' ) . '</p></div>';
		}
	}

	/**
	 * Check if this gateway is enabled
	 */
	public function is_available() {
		if ( 'yes' === $this->enabled ) {
			if ( ! $this->testmode && is_checkout() && ! is_ssl() ) {
				return false;
			}
			if ( (! $this->secret_key || ! $this->publishable_key ) && (! $this->secret_key2 || ! $this->publishable_key2 ) && (! $this->secret_key3 || ! $this->publishable_key3 ))  {
				return false;
			}
			return true;
		}
		return false;
	}

	/**
     * Initialise Gateway Settings Form Fields
     */
	public function init_form_fields() {
        $this->form_fields = apply_filters( 'wc_nuvei_settings',
            array(
                'enabled' => array(
                    'title'       => __( 'Enable/Disable', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable Nuvei', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => '',
                    'default'     => 'no',
                ),
                'title' => array(
                    'title'       => __( 'Title', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( 'This controls the title which the user sees during checkout.', 'woocommerce-gateway-nuvei' ),
                    'default'     => __( 'Debit or Credit Card (Nuvei)', 'woocommerce-gateway-nuvei' ),
                    'desc_tip'    => true,
                    'custom_attributes' => array('readonly' => 'readonly')
                ),
                'description' => array(
                    'title'       => __( 'Description', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( 'This controls the description which the user sees during checkout.', 'woocommerce-gateway-nuvei' ),
                    'default'     => __( 'Pay with your debit or credit card via Nuvei.', 'woocommerce-gateway-nuvei' ),
                    'desc_tip'    => true,
                    'custom_attributes' => array('readonly' => 'readonly')
                ),
                'testmode' => array(
                    'title'       => __( 'Test mode', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable Test Mode', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( 'Place the payment gateway in test mode using test API keys.', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'yes',
                    'desc_tip'    => true,
                ),
                'avs' => array(
                    'title'       => __( 'AVS', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable AVS', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( 'Make sure AVS is enabled on your terminal.', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                '3ds' => array(
                    'title'       => __( '3DS', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable 3DS', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( 'Make sure 3DS is enabled on your terminal.', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                'mpi_receipt_url' => array(
                    'title'       => __( 'MPI Receipt URL', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( '', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( 'Set this value in your Selfcare Settings > Terminal', 'woocommerce-gateway-nuvei' ),
                    'default'     => get_permalink( wc_get_page_id( 'checkout' ) ),
                    'desc_tip'    => true,
                    'custom_attributes' => array('readonly' => 'readonly')
                ),

                'background_validation' => array(
                    'title'       => __( 'Background Validation', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable validation', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( 'Make sure Validation is enabled on your terminal.', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                'background_validation_url' => array(
                    'title'       => __( 'Validation URL', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( '', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( 'Set this value in your Selfcare Settings > Terminal', 'woocommerce-gateway-nuvei' ),
                    'default'     => wc_get_cart_url(),
                    'desc_tip'    => true,
                    'custom_attributes' => array('readonly' => 'readonly')
                ),

                'test_publishable_key' => array(
                    'title'       => __( 'Test Primary Terminal Id', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'test_secret_key' => array(
                    'title'       => __( 'Test Primary Shared Secret', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'test_currency' => array(
                    'title'       => __( 'Test Primary Currency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'Euro',
                    'desc_tip'    => true,
                    'options'     => array(
                        'EUR' => __( 'Euro', 'woocommerce-gateway-nuvei' ),
                        'GBP'   => __( 'Sterling', 'woocommerce-gateway-nuvei' ),
                        'USD'   => __( 'US Dollar', 'woocommerce-gateway-nuvei' ),
                        'CAD'   => __( 'Canadian Dollar', 'woocommerce-gateway-nuvei' ),
                        'AUD'   => __( 'Australian Dollar', 'woocommerce-gateway-nuvei' ),
                        'DKK'   => __( 'Danish Krone', 'woocommerce-gateway-nuvei' ),
                        'SEK'   => __( 'Swedish Krona', 'woocommerce-gateway-nuvei' ),
                        'NOK'   => __( 'Norwegian Krone', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
                'test_multicurrency' => array(
                    'title'       => __( 'Test multicurrency', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable multicurrency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                'test_publishable_key2' => array(
                    'title'       => __( 'Test Second Terminal Id', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'test_secret_key2' => array(
                    'title'       => __( 'Test Second Shared Secret', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'test_currency2' => array(
                    'title'       => __( 'Test Second Currency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'Euro',
                    'desc_tip'    => true,
                    'options'     => array(
                        'EUR' => __( 'Euro', 'woocommerce-gateway-nuvei' ),
                        'GBP'   => __( 'Sterling', 'woocommerce-gateway-nuvei' ),
                        'USD'   => __( 'US Dollar', 'woocommerce-gateway-nuvei' ),
                        'CAD'   => __( 'Canadian Dollar', 'woocommerce-gateway-nuvei' ),
                        'AUD'   => __( 'Australian Dollar', 'woocommerce-gateway-nuvei' ),
                        'DKK'   => __( 'Danish Krone', 'woocommerce-gateway-nuvei' ),
                        'SEK'   => __( 'Swedish Krona', 'woocommerce-gateway-nuvei' ),
                        'NOK'   => __( 'Norwegian Krone', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
                'test_multicurrency2' => array(
                    'title'       => __( 'Test Second multicurrency', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable multicurrency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                'test_publishable_key3' => array(
                    'title'       => __( 'Test Third Terminal Id', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'test_secret_key3' => array(
                    'title'       => __( 'Test Third Shared Secret', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'test_currency3' => array(
                    'title'       => __( 'Test Third Currency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'Euro',
                    'desc_tip'    => true,
                    'options'     => array(
                        'EUR' => __( 'Euro', 'woocommerce-gateway-nuvei' ),
                        'GBP'   => __( 'Sterling', 'woocommerce-gateway-nuvei' ),
                        'USD'   => __( 'US Dollar', 'woocommerce-gateway-nuvei' ),
                        'CAD'   => __( 'Canadian Dollar', 'woocommerce-gateway-nuvei' ),
                        'AUD'   => __( 'Australian Dollar', 'woocommerce-gateway-nuvei' ),
                        'DKK'   => __( 'Danish Krone', 'woocommerce-gateway-nuvei' ),
                        'SEK'   => __( 'Swedish Krona', 'woocommerce-gateway-nuvei' ),
                        'NOK'   => __( 'Norwegian Krone', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
                'test_multicurrency3' => array(
                    'title'       => __( 'Test Third multicurrency', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable multicurrency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),

                'live_publishable_key' => array(
                    'title'       => __( 'Live Primary Terminal Id', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'live_secret_key' => array(
                    'title'       => __( 'Live Primary Shared Secret', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'live_currency' => array(
                    'title'       => __( 'Live Primary Currency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'Euro',
                    'desc_tip'    => true,
                    'options'     => array(
                        'EUR' => __( 'Euro', 'woocommerce-gateway-nuvei' ),
                        'GBP'   => __( 'Sterling', 'woocommerce-gateway-nuvei' ),
                        'USD'   => __( 'US Dollar', 'woocommerce-gateway-nuvei' ),
                        'CAD'   => __( 'Canadian Dollar', 'woocommerce-gateway-nuvei' ),
                        'AUD'   => __( 'Australian Dollar', 'woocommerce-gateway-nuvei' ),
                        'DKK'   => __( 'Danish Krone', 'woocommerce-gateway-nuvei' ),
                        'SEK'   => __( 'Swedish Krona', 'woocommerce-gateway-nuvei' ),
                        'NOK'   => __( 'Norwegian Krone', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
                'live_multicurrency' => array(
                    'title'       => __( 'Live multicurrency', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable multicurrency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                'live_publishable_key2' => array(
                    'title'       => __( 'Live Second Terminal Id', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'live_secret_key2' => array(
                    'title'       => __( 'Live Second Shared Secret', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'live_currency2' => array(
                    'title'       => __( 'Live Second Currency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'Euro',
                    'desc_tip'    => true,
                    'options'     => array(
                        'EUR' => __( 'Euro', 'woocommerce-gateway-nuvei' ),
                        'GBP'   => __( 'Sterling', 'woocommerce-gateway-nuvei' ),
                        'USD'   => __( 'US Dollar', 'woocommerce-gateway-nuvei' ),
                        'CAD'   => __( 'Canadian Dollar', 'woocommerce-gateway-nuvei' ),
                        'AUD'   => __( 'Australian Dollar', 'woocommerce-gateway-nuvei' ),
                        'DKK'   => __( 'Danish Krone', 'woocommerce-gateway-nuvei' ),
                        'SEK'   => __( 'Swedish Krona', 'woocommerce-gateway-nuvei' ),
                        'NOK'   => __( 'Norwegian Krone', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
                'live_multicurrency2' => array(
                    'title'       => __( 'Live Second multicurrency', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable multicurrency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),
                'live_publishable_key3' => array(
                    'title'       => __( 'Live Third Terminal Id', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'live_secret_key3' => array(
                    'title'       => __( 'Live Third Shared Secret', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'text',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'live_currency3' => array(
                    'title'       => __( 'Live Third Currency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'Euro',
                    'desc_tip'    => true,
                    'options'     => array(
                        'EUR' => __( 'Euro', 'woocommerce-gateway-nuvei' ),
                        'GBP'   => __( 'Sterling', 'woocommerce-gateway-nuvei' ),
                        'USD'   => __( 'US Dollar', 'woocommerce-gateway-nuvei' ),
                        'CAD'   => __( 'Canadian Dollar', 'woocommerce-gateway-nuvei' ),
                        'AUD'   => __( 'Australian Dollar', 'woocommerce-gateway-nuvei' ),
                        'DKK'   => __( 'Danish Krone', 'woocommerce-gateway-nuvei' ),
                        'SEK'   => __( 'Swedish Krona', 'woocommerce-gateway-nuvei' ),
                        'NOK'   => __( 'Norwegian Krone', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
                'live_multicurrency3' => array(
                    'title'       => __( 'Live Third multicurrency', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Enable multicurrency', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'no',
                    'desc_tip'    => true,
                ),

                /*
                'capture' => array(
                    'title'       => __( 'Capture', 'woocommerce-gateway-nuvei' ),
                    'label'       => __( 'Capture charge immediately', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'checkbox',
                    'description' => __( 'Whether or not to immediately capture the charge. When unchecked, the charge issues an authorization and will need to be captured later. Uncaptured charges expire in 7 days.', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'yes',
                    'desc_tip'    => true,
                ),*/
                'nuvei_integration_type' => array(
                    'title'       => __( 'Integration type', 'woocommerce-gateway-nuvei' ),
                    'type'        => 'select',
                    'class'       => '',
                    'description' => __( '', 'woocommerce-gateway-nuvei' ),
                    'default'     => 'XML',
                    'desc_tip'    => true,
                    'options'     => array(
                        'xml' => __( 'XML', 'woocommerce-gateway-nuvei' ),
                        'hpp'   => __( 'HPP', 'woocommerce-gateway-nuvei' ),
                    ),
                ),
            )
        );
    }

	public function get_terminal_settings() {
	    $currency = get_woocommerce_currency();

        if($this->currency == $currency && $this->publishable_key && $this->secret_key) {
            $terminalId = $this->publishable_key;
            $customFieldLanguage = 'en-US';
            $secret = $this->secret_key;

            $XmlTerminalFeatures = new NuveiGatewayXmlTerminalFeaturesRequest($terminalId, $customFieldLanguage);

            $serverUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
            $response = $XmlTerminalFeatures->ProcessRequestToGateway($secret, $serverUrl);
            $terminalFeatures = $response->getSettings();

            $this->terminal_settings = $terminalFeatures;
        }
        if($this->currency2 == $currency && $this->publishable_key2 && $this->secret_key2) {
            $terminalId = $this->publishable_key2;
            $customFieldLanguage = 'en-US';
            $secret = $this->secret_key2;

            $XmlTerminalFeatures = new NuveiGatewayXmlTerminalFeaturesRequest($terminalId, $customFieldLanguage);

            $serverUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
            $response = $XmlTerminalFeatures->ProcessRequestToGateway($secret, $serverUrl);
            $terminalFeatures = $response->getSettings();

            $this->terminal_settings = $terminalFeatures;
        }
        if($this->currency3 == $currency && $this->publishable_key3 && $this->secret_key3) {
            $terminalId = $this->publishable_key3;
            $customFieldLanguage = 'en-US';
            $secret = $this->secret_key3;

            $XmlTerminalFeatures = new NuveiGatewayXmlTerminalFeaturesRequest($terminalId, $customFieldLanguage);

            $serverUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
            $response = $XmlTerminalFeatures->ProcessRequestToGateway($secret, $serverUrl);
            $terminalFeatures = $response->getSettings();

            $this->terminal_settings = $terminalFeatures;
        }
    }

	/**
	 * Payment form on checkout page
	 */
	public function payment_fields() {
		$user                 = wp_get_current_user();
		$display_tokenization = $this->supports( 'tokenization' ) && is_checkout() && $this->saved_cards;
		$total                = WC()->cart->total;

		// If paying from order, we need to get total from order not cart.
		if ( isset( $_GET['pay_for_order'] ) && ! empty( $_GET['key'] ) ) {
			$order = wc_get_order( wc_get_order_id_by_order_key( wc_clean( $_GET['key'] ) ) );
			$total = $order->get_total();
		}

		if ( $user->ID ) {
			$user_email = get_user_meta( $user->ID, 'billing_email', true );
			$user_email = $user_email ? $user_email : $user->user_email;
		} else {
			$user_email = '';
		}

		if ( is_add_payment_method_page() ) {
			$pay_button_text = __( 'Add Card', 'woocommerce-gateway-nuvei' );
			$total        = '';
		} else {
			$pay_button_text = '';
		}


		echo '<div
			id="nuvei-payment-data"
			data-panel-label="' . esc_attr( $pay_button_text ) . '"
			data-description=""
			data-name="' . esc_attr( $this->statement_descriptor ) . '"
			data-image="' . esc_attr( $this->nuvei_checkout_image ) . '"
			data-locale="' . esc_attr( $this->nuvei_checkout_locale ? $this->nuvei_checkout_locale : 'en' ) . '"
			data-allow-remember-me="' . esc_attr( $this->saved_cards ? 'true' : 'false' ) . '">';

		if ( $this->description ) {
			echo apply_filters( 'wc_nuvei_description', wpautop( wp_kses_post( $this->description ) ) );
		}

		if ( $display_tokenization ) {
			$this->tokenization_script();
			$this->saved_payment_methods();
		}

		if ( $this->nuvei_integration_type=='xml') {
			$this->form();

			if ( apply_filters( 'wc_nuvei_display_save_payment_method_checkbox', $display_tokenization ) ) {
				$this->save_payment_method_checkbox();
			}
		}

		echo '</div>';
	}

	/**
	 * Localize Nuvei messages based on code
	 *
	 * @since 3.0.6
	 * @version 3.0.6
	 * @return array
	 */
	public function get_localized_messages() {
		return apply_filters( 'wc_nuvei_localized_messages', array(
			'invalid_number'        => __( 'The card number is not a valid credit card number.', 'woocommerce-gateway-nuvei' ),
			'invalid_expiry_month'  => __( 'The card\'s expiration month is invalid.', 'woocommerce-gateway-nuvei' ),
			'invalid_expiry_year'   => __( 'The card\'s expiration year is invalid.', 'woocommerce-gateway-nuvei' ),
			'invalid_cvc'           => __( 'The card\'s security code is invalid.', 'woocommerce-gateway-nuvei' ),
			'incorrect_number'      => __( 'The card number is incorrect.', 'woocommerce-gateway-nuvei' ),
			'expired_card'          => __( 'The card has expired.', 'woocommerce-gateway-nuvei' ),
			'incorrect_cvc'         => __( 'The card\'s security code is incorrect.', 'woocommerce-gateway-nuvei' ),
			'incorrect_zip'         => __( 'The card\'s zip code failed validation.', 'woocommerce-gateway-nuvei' ),
			'card_declined'         => __( 'The card was declined.', 'woocommerce-gateway-nuvei' ),
			'missing'               => __( 'There is no card on a customer that is being charged.', 'woocommerce-gateway-nuvei' ),
			'processing_error'      => __( 'An error occurred while processing the card.', 'woocommerce-gateway-nuvei' ),
			'invalid_request_error' => __( 'Could not find payment information.', 'woocommerce-gateway-nuvei' ),
		) );
	}

	/**
	 * Load admin scripts.
	 *
	 * @since 3.1.0
	 * @version 3.1.0
	 */
	public function admin_scripts() {
		if ( 'woocommerce_page_wc-settings' !== get_current_screen()->id ) {
			return;
		}


		wp_enqueue_script( 'nuvei_gateway_admin', plugins_url( 'assets/js/nuvei-admin.js', Nuvei_Gateway_for_WC_MAIN_FILE ), array(), Nuvei_Gateway_for_WC_VERSION, true );

		$nuvei_admin_params = array(
			'localized_messages' => array(
				'not_valid_live_key_msg' => __( 'This is not a valid live key. Live keys start with "sk_live_" and "pk_live_".', 'woocommerce-gateway-nuvei' ),
				'not_valid_test_key_msg' => __( 'This is not a valid test key. Test keys start with "sk_test_" and "pk_test_".', 'woocommerce-gateway-nuvei' ),
				're_verify_button_text'  => __( 'Re-verify Domain', 'woocommerce-gateway-nuvei' ),
				'missing_secret_key'     => __( 'Missing Secret Key. Please set the secret key field above and re-try.', 'woocommerce-gateway-nuvei' ),
			),
			'ajaxurl'            => admin_url( 'admin-ajax.php' )
		);

		wp_localize_script( 'nuvei_gateway_admin', 'nuvei_gateway_admin_params', apply_filters( '_nuvei_gateway_admin_params', $nuvei_admin_params ) );
	}

	/**
	 * payment_scripts function.
	 *
	 * Outputs scripts used for nuvei payment
	 *
	 * @access public
	 */
	public function payment_scripts() {
		if ( ! is_cart() && ! is_checkout() && ! isset( $_GET['pay_for_order'] ) && ! is_add_payment_method_page() ) {
			return;
		}

		if ( $this->nuvei_integration_type == 'hpp' ) {
			wp_enqueue_script( 'woocommerce_nuvei', plugins_url( 'assets/js/nuvei-hpp.js', Nuvei_Gateway_for_WC_MAIN_FILE ), '', Nuvei_Gateway_for_WC_VERSION, true );
		} else {
			wp_enqueue_script( 'ccValidator', plugins_url( 'assets/js/jquery.creditCardValidator.js', Nuvei_Gateway_for_WC_MAIN_FILE ), '', Nuvei_Gateway_for_WC_VERSION, true );
			wp_enqueue_script( 'woocommerce_nuvei', plugins_url( 'assets/js/nuvei-xml.js', Nuvei_Gateway_for_WC_MAIN_FILE ), array( 'jquery-payment', 'ccValidator'), Nuvei_Gateway_for_WC_VERSION, true );
		}

		$nuvei_params = array(
			'key'                  => $this->publishable_key,
			'i18n_terms'           => __( 'Please accept the terms and conditions first', 'woocommerce-gateway-nuvei' ),
			'i18n_required_fields' => __( 'Please fill in required checkout fields first', 'woocommerce-gateway-nuvei' ),
		);

		// If we're on the pay page we need to pass nuvei.js the address of the order.
		if ( isset( $_GET['pay_for_order'] ) && 'true' === wc_clean(wp_unslash($_GET['pay_for_order'])) ) {
			$order_id = wc_get_order_id_by_order_key( urldecode( $_GET['key'] ) );
			$order    = wc_get_order( $order_id );

			$nuvei_params['billing_first_name'] = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_first_name : $order->get_billing_first_name();
			$nuvei_params['billing_last_name']  = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_last_name : $order->get_billing_last_name();
			$nuvei_params['billing_address_1']  = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_1 : $order->get_billing_address_1();
			$nuvei_params['billing_address_2']  = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_2 : $order->get_billing_address_2();
			$nuvei_params['billing_state']      = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_state : $order->get_billing_state();
			$nuvei_params['billing_city']       = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_city : $order->get_billing_city();
			$nuvei_params['billing_postcode']   = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_postcode : $order->get_billing_postcode();
			$nuvei_params['billing_country']    = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_country : $order->get_billing_country();
		}

		$nuvei_params['no_prepaid_card_msg']                     = __( 'Sorry, we\'re not accepting prepaid cards at this time.', 'woocommerce-gateway-nuvei' );
		$nuvei_params['allow_prepaid_card']                      = apply_filters( 'wc_nuvei_allow_prepaid_card', true ) ? 'yes' : 'no';
		$nuvei_params['nuvei_checkout_require_billing_address'] = apply_filters( 'wc_nuvei_checkout_require_billing_address', false ) ? 'yes' : 'no';

		// merge localized messages to be use in JS
		$nuvei_params = array_merge( $nuvei_params, $this->get_localized_messages() );

		wp_localize_script( 'woocommerce_nuvei', 'wc_nuvei_params', apply_filters( 'wc_nuvei_params', $nuvei_params ) );
	}

    public function getCreditCardType($str)
    {
        if (empty($str)) {
            return false;
        }

        $matchingPatterns = [
            'visa' => '/^4[0-9]{12}(?:[0-9]{3})?$/',
            'mastercard' => '/^(5[1-5][0-9]{14})|(2[2-7][0-9]{14})$/',
            'laser' => '/^(6304|6706|6709|6771)[0-9]{12,15}$/',
            'maestro' => '/^(50|5[6-9]|6[0-9])[0-9]{10,17}$/',
            'amex' => '/^3[47][0-9]{13}$/',
            'diners' => '/^3(?:0[0-5]|[68][0-9])[0-9]{11}$/',
            'discover' => '/^6(?:011|5[0-9]{2})[0-9]{12}$/',
            'jcb' => '/^(?:2131|1800|35\d{3})\d{11}$/',
            'UNKNOWN' => '/^((?:4[0-9]{12}(?:[0-9]{3})?|5[1-5][0-9]{14}|6(?:011|5[0-9][0-9])[0-9]{12}|3[47][0-9]{13}|3(?:0[0-5]|[68][0-9])[0-9]{11}|(?:2131|1800|35\d{3})\d{11}))|([0-9]{16,19})$/'
        ];

        foreach ($matchingPatterns as $key=>$pattern) {
            if (preg_match($pattern, $str)) {
                return strtoupper($key);
            }
        }
    }

	/**
	 * Process the payment
	 *
	 * @param int  $order_id Reference.
	 * @param bool $retry Should we retry on fail.
	 * @param bool $force_customer Force user creation.
	 *
	 * @throws Exception If payment will not be accepted.
	 *
	 * @return array|void
	 */
	public function process_payment( $order_id, $retry = true, $force_customer = false ) {
		if ( $this->nuvei_integration_type=='hpp') {
			$order = wc_get_order($order_id);

			$dateTime = date('d-m-Y:H:i:s:000');
			$receiptPageUrl = $this->get_return_url($order);

			$currency = wc_clean(wp_unslash($order->get_currency()));
            if($currency == $this->currency && $this->publishable_key && $this->secret_key) {
                $terminalId = $this->publishable_key;
                $secret = $this->secret_key;
                $multicurrency = $this->multicurrency;
            } else if($currency == $this->currency2 && $this->publishable_key2 && $this->secret_key2) {
				$terminalId = $this->publishable_key2;
				$secret = $this->secret_key2;
				$multicurrency = $this->multicurrency2;
			} else if($currency == $this->currency3 && $this->publishable_key3 && $this->secret_key3) {
					$terminalId = $this->publishable_key3;
					$secret = $this->secret_key3;
					$multicurrency = $this->multicurrency3;
			}
			if(!$terminalId) {
                wc_add_notice( 'Payment plugin misconfigured: terminal for current currency not available.', 'error' );
                return;
            }

            $HPP_VARS = '';
            if($this->avs) {
                $address1 = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_1 : $order->get_billing_address_1();
                $address2 = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_2 : $order->get_billing_address_2();
                $postcode = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_postcode : $order->get_billing_postcode();
                $city = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_city : $order->get_billing_city();
                $region = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_state : $order->get_billing_state();
                $country = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_country : $order->get_billing_country();
                $email = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_email : $order->get_billing_email();
                $phone = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_phone : $order->get_billing_phone();
                $HPP_VARS = '&ADDRESS1='.$address1.'&ADDRESS2='.$address2.'&POSTCODE='.$postcode.'&CITY='.$city.'&REGION='.$region.'&COUNTRY='.$country.'&EMAIL='.$email.'&PHONE='.$phone;
            }
            $firstName = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_first_name : $order->get_billing_first_name();
            $lastName = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_last_name : $order->get_billing_last_name();

			$amount = number_format(wc_clean(wp_unslash($order->get_total())), 2, '.', '');
			$hash = md5($terminalId . $order_id . ($multicurrency=='yes' ? $currency : '') . $amount . $dateTime . $receiptPageUrl . $secret);

			$redirectUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/paymentpage':'https://payments.nuvei.com/merchant/paymentpage';
			$redirectUrl .= '?TERMINALID=' . $terminalId . '&ORDERID=' . $order_id . '&CARDHOLDERNAME=' . $firstName .' '.$lastName . '&CURRENCY=' . wc_clean(wp_unslash($order->get_currency())) . '&AMOUNT=' . $amount . '&DATETIME=' . $dateTime . '&RECEIPTPAGEURL=' . urlencode($receiptPageUrl) . $HPP_VARS . '&HASH=' . $hash;

			return array(
                'result' => 'success',
                'redirect' => $redirectUrl,
            );
		} else if ( $this->nuvei_integration_type=='xml') {
            session_start();
            $csrfToken = wc_clean(wp_unslash($_POST['csrf-token']));
		    if(!hash_equals($_SESSION['CSRF-Token'], $csrfToken)) {
                unset($_SESSION['CSRF-Token']);

                return array(
                    'result' => 'success',
                    'redirect' => wc_get_checkout_url(),
                );
            }
            unset($_SESSION['CSRF-Token']);

			$order = wc_get_order($order_id);

			$currency = wc_clean(wp_unslash($order->get_currency()));
            if($currency == $this->currency && $this->publishable_key && $this->secret_key) {
                $terminalId = $this->publishable_key;
                $secret = $this->secret_key;
                $multicurrency = $this->multicurrency;
            } else if($currency == $this->currency2 && $this->publishable_key2 && $this->secret_key2) {
				$terminalId = $this->publishable_key2;
				$secret = $this->secret_key2;
				$multicurrency = $this->multicurrency2;
			} else if($currency == $this->currency3 && $this->publishable_key3 && $this->secret_key3) {
				$terminalId = $this->publishable_key3;
				$secret = $this->secret_key3;
				$multicurrency = $this->multicurrency3;
			}
            if(!$terminalId) {
                wc_add_notice( 'Payment plugin misconfigured: terminal for current currency not available.', 'error' );
                return;
            }

			# Set up the authorisation object
			$amount = number_format(wc_clean(wp_unslash($order->get_total())), 2, '.', '');
			$cardNumber = str_replace(' ', '', wc_clean(wp_unslash($_POST['card-number'])));
			$cardType = $this->getCreditCardType($cardNumber);
			$cardHolder = wc_clean(wp_unslash($_POST['card-name']));
			$cardExpiry = wc_clean(wp_unslash($_POST['card-expiry']));
			$cvv = wc_clean(wp_unslash($_POST['cvc']));

			$adminEmail = '';
			$adminPhone = '';

			if($this->enabled3ds) {
			    //save card details in WC session to be processed after 3DS response
                $cardDetails['testmode'] = $this->testmode;
                $cardDetails['avs'] = $this->avs;
                $cardDetails['multicurrency'] = $multicurrency;
                $cardDetails['terminalId'] = $terminalId;
                $cardDetails['secret'] = $secret;
                $cardDetails['currency'] = $currency;
                $cardDetails['amount'] = $amount;
                $cardDetails['cardNumber'] = $cardNumber;
                $cardDetails['cardHolder'] = $cardHolder;
                $cardDetails['cardExpiry'] = $cardExpiry;
                $cardDetails['cardType'] = $cardType;
                $cardDetails['cvv'] = $cvv;
                WC()->session->set( 'wc_gateway_nuvei_'.$order_id, $cardDetails);
                //end save


                $dateTime = date('d-m-Y:H:i:s:000');
                $hash = md5($terminalId . $order_id . $cardNumber . $cardExpiry . $cardType . $amount . $dateTime . $secret);

                $redirectUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/mpi':'https://payments.nuvei.com/merchant/mpi';
                //$redirectUrl .= '?TERMINALID=' . $terminalId . '&CARDHOLDERNAME=' . $cardHolder . '&CARDNUMBER=' . $cardNumber . '&CARDEXPIRY=' . $cardExpiry . '&CARDTYPE=' . $cardType . '&AMOUNT=' . number_format($amount, 2, '.', '') . '&CURRENCY=' . $currency . '&ORDERID=' . $order_id . '&CVV='. $cvv . '&DATETIME=' . $dateTime . '&HASH=' . $hash;

                $request3ds = array();
                $request3ds['TERMINALID'] = $terminalId;
                $request3ds['CARDHOLDERNAME'] = $cardHolder;
                $request3ds['CARDNUMBER'] = $cardNumber;
                $request3ds['CARDEXPIRY'] = $cardExpiry;
                $request3ds['CARDTYPE'] = $cardType;
                $request3ds['AMOUNT'] = number_format($amount, 2, '.', '');
                $request3ds['CURRENCY'] = $currency;
                $request3ds['ORDERID'] = $order_id;
                $request3ds['CVV'] = $cvv;
                $request3ds['DATETIME'] = $dateTime;
                $request3ds['HASH'] = $hash;

                $request3ds['redirectUrl'] = $redirectUrl;

                WC()->session->set( 'request3ds', $request3ds);

                return array(
                    'result' => 'success',
                    'redirect' => wc_get_checkout_url(),
                );
            } else {
                $XmlAuthRequest = new NuveiGatewayXmlAuthRequest($terminalId, $order_id, $currency, $amount, $cardNumber, $cardType);

                if ($cardType != "SECURECARD") $XmlAuthRequest->SetNonSecureCardCardInfo($cardExpiry, $cardHolder);
                if ($cvv != "") $XmlAuthRequest->SetCvv($cvv);
                if ($multicurrency=='yes') $XmlAuthRequest->SetMultiCur();


                if(true || $this->avs) { //always send AVS data on xml requests and let gateway decide if will use it or not
                    $address1 = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_1 : $order->get_billing_address_1();
                    $address2 = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_address_2 : $order->get_billing_address_2();
                    $postcode = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_postcode : $order->get_billing_postcode();
                    $city = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_city : $order->get_billing_city();
                    $region = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_state : $order->get_billing_state();
                    $country = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_country : $order->get_billing_country();
                    $email = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_email : $order->get_billing_email();
                    $phone = version_compare( WC_VERSION, '3.0.0', '<' ) ? $order->billing_phone : $order->get_billing_phone();


                    $XmlAuthRequest->SetAvs($address1, $address2, $postcode);
                    $XmlAuthRequest->SetCity($city);
                    if($region != "") $XmlAuthRequest->SetRegion($region);
                    $XmlAuthRequest->SetCountry($country);
                    $XmlAuthRequest->SetEmail($email);
                    $XmlAuthRequest->SetPhone($phone);
                }

                # Perform the online authorisation and read in the result
                $serverUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
                $response = $XmlAuthRequest->ProcessRequestToGateway($secret, $serverUrl);

                $expectedResponseHash = md5($terminalId . $response->UniqueRef() . ($multicurrency=='yes' ? $currency : '')  . $amount . $response->DateTime() . $response->ResponseCode() . $response->ResponseText() . $secret);
                $isHashCorrect = ($expectedResponseHash == $response->Hash());

                $nuveiResponse = '';

                if ($response->IsError()) $nuveiResponse .= 'AN ERROR OCCURED! Your transaction was not processed. Error details: ' . $response->ErrorString();
                elseif ($isHashCorrect) {
                    switch ($response->ResponseCode()) {
                        case "A" :    # -- If using local database, update order as Authorised.
                            $nuveiResponse .= 'Payment Processed successfully. Thanks you for your order.';
                            $uniqueRef = $response->UniqueRef();
                            $responseText = $response->ResponseText();
                            $approvalCode = $response->ApprovalCode();
                            $avsResponse = $response->AvsResponse();
                            $cvvResponse = $response->CvvResponse();
                            break;
                        case "R" :
                        case "D" :
                        case "C" :
                        case "S" :
                        default  :    # -- If using local database, update order as declined/failed --
                            $nuveiResponse .= 'PAYMENT DECLINED! Please try again with another card. Bank response: ' . $response->ResponseText();
                    }
                } else {
                    $nuveiResponse .= 'PAYMENT FAILED: INVALID RESPONSE HASH. Please contact ' . $adminEmail . ' or call ' . $adminPhone . ' to clarify if you will get charged for this order.';
                    if ($response->UniqueRef()) $nuveiResponse .= 'Please quote Nuvei Terminal ID: ' . $terminalId . ', and Unique Reference: ' . $response->UniqueRef() . ' when mailing or calling.';
                }

                $receiptPageUrl = $this->get_return_url($order);

                if (!$response->IsError() && $isHashCorrect && $response->ResponseCode() == 'A') {
                    $order->update_status( apply_filters( 'woocommerce_gateway_nuvei_process_payment_order_status', 'processing', $order ), __( 'Payment successfully processed #'. $response->UniqueRef(), 'woocommerce' ) );

                    $order->set_transaction_id( $response->UniqueRef() );

                    $order->save();

                    WC()->mailer()->customer_invoice( $order );

                    wc_reduce_stock_levels( $order_id );

                    WC()->cart->empty_cart();

                    return array(
                        'result' => 'success',
                        'redirect' => $receiptPageUrl,
                    );
                }
                else {
                    wc_add_notice( $nuveiResponse, 'error' );
                }
            }
		}

	}

	/**
     * Refund a charge
     * @param  int $order_id
     * @param  float $amount
     * @return bool
     */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
        $order = wc_get_order( $order_id );

        if ( ! $order || ! $order->get_transaction_id() || $amount <= 0 ) {
            if($amount <= 0)
                throw new Exception("Amount should be positive.");
            return false;
        }

        $reason = 'Refund reason: '.($reason == ''?'none':$reason);
        $amount = number_format(wc_clean(wp_unslash($amount)), 2, '.', '');

        $currency = wc_clean(wp_unslash($order->get_currency()));

        if($currency == $this->currency && $this->publishable_key && $this->secret_key) {
            $terminalId = $this->publishable_key;
            $secret = $this->secret_key;
        } else if($currency == $this->currency2 && $this->publishable_key2 && $this->secret_key2) {
            $terminalId = $this->publishable_key2;
            $secret = $this->secret_key2;
        } else if($currency == $this->currency3 && $this->publishable_key3 && $this->secret_key3) {
            $terminalId = $this->publishable_key3;
            $secret = $this->secret_key3;
        }
        if(!$terminalId) {
            throw new Exception( 'Payment plugin misconfigured: terminal for current currency not available.');
            return false;
        }

        $uniqueRef = $order->get_transaction_id();
        $operator = 'Woocommerce payment plugin';

        $XmlRefundRequest = new NuveiGatewayXmlRefundRequest($terminalId, $order_id, $amount, $operator, $reason);
        $XmlRefundRequest->SetUniqueRef($uniqueRef);

        $serverUrl = $this->testmode == 'yes' ?'https://testpayments.nuvei.com/merchant/xmlpayment':'https://payments.nuvei.com/merchant/xmlpayment';
        $response = $XmlRefundRequest->ProcessRequestToGateway($secret, $serverUrl);

        $expectedResponseHash = md5($terminalId.$response->UniqueRef().$amount.$response->DateTime() . $response->ResponseCode() . $response->ResponseText().$secret);

        $responseBody = '';
        if ($response->IsError()) {
            $responseBody .= 'AN ERROR OCCURED! Error details: ' . $response->ErrorString();
        }
        else {
            if ($expectedResponseHash == $response->Hash() && $response->ResponseCode() == 'A') {
                $order->add_order_note( 'Refund processed correctly #'. $response->UniqueRef() ."\n". $reason );
                $order->save();
                return true;
            } else {
                $responseBody .= 'AN ERROR OCCURED! Error details: ' . $response->ResponseText();
            }
        }

        if($responseBody) {
            throw new Exception($responseBody);
        }

        return true;
    }

	/**
     * Sends the failed order email to admin
     *
     * @version 3.1.0
     * @since 3.1.0
     * @param int $order_id
     * @return null
     */
	public function send_failed_order_email( $order_id ) {
        $emails = WC()->mailer()->get_emails();
        if ( ! empty( $emails ) && ! empty( $order_id ) ) {
            $emails['WC_Email_Failed_Order']->trigger( $order_id );
        }
    }

	/**
	 * Logs
	 *
	 * @since 3.1.0
	 * @version 3.1.0
	 *
	 * @param string $message
	 */
	public function log( $message ) {
		if ( $this->logging ) {
			Nuvei_Gateway_for_WC::log( $message );
		}
	}
}
